"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteDataStream = deleteDataStream;
exports.updateDataStreamsLifecycle = updateDataStreamsLifecycle;
exports.updateOrRolloverDataStream = updateOrRolloverDataStream;
exports.upsertDataStream = upsertDataStream;
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _retry = require("../helpers/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function upsertDataStream({
  esClient,
  name,
  logger
}) {
  const dataStreamExists = await esClient.indices.exists({
    index: name
  });
  if (dataStreamExists) {
    return;
  }
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.createDataStream({
      name
    }), {
      logger
    });
    logger.debug(() => `Installed data stream: ${name}`);
  } catch (error) {
    logger.error(`Error creating data stream: ${error.message}`);
    throw error;
  }
}
async function deleteDataStream({
  esClient,
  name,
  logger
}) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.deleteDataStream({
      name
    }, {
      ignore: [404]
    }), {
      logger
    });
  } catch (error) {
    logger.error(`Error deleting data stream: ${error.message}`);
    throw error;
  }
}
async function updateOrRolloverDataStream({
  esClient,
  name,
  logger
}) {
  const dataStreams = await esClient.indices.getDataStream({
    name
  });
  for (const dataStream of dataStreams.data_streams) {
    // simulate index and try to patch the write index
    // if that doesn't work, roll it over
    const simulatedIndex = await esClient.indices.simulateIndexTemplate({
      name
    });
    const writeIndex = dataStream.indices.at(-1);
    if (!writeIndex) {
      continue;
    }
    try {
      // Apply blocklist to avoid changing settings we don't want to
      const simulatedIndexSettings = (0, _lodash.omit)(simulatedIndex.template.settings, ['index.codec', 'index.mapping.ignore_malformed', 'index.mode', 'index.logsdb.sort_on_host_name']);
      await (0, _retry.retryTransientEsErrors)(() => Promise.all([esClient.indices.putMapping({
        index: writeIndex.index_name,
        properties: simulatedIndex.template.mappings.properties
      }), esClient.indices.putSettings({
        index: writeIndex.index_name,
        settings: simulatedIndexSettings
      })]), {
        logger
      });
    } catch (error) {
      if (typeof error.message !== 'string' || !error.message.includes('illegal_argument_exception')) {
        throw error;
      }
      try {
        await (0, _retry.retryTransientEsErrors)(() => esClient.indices.rollover({
          alias: dataStream.name
        }), {
          logger
        });
        logger.debug(() => `Rolled over data stream: ${dataStream.name}`);
      } catch (rolloverError) {
        logger.error(`Error rolling over data stream: ${error.message}`);
        throw error;
      }
    }
  }
}
async function updateDataStreamsLifecycle({
  esClient,
  logger,
  names,
  lifecycle,
  isServerless
}) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.putDataLifecycle({
      name: names,
      data_retention: (0, _streamsSchema.isDslLifecycle)(lifecycle) ? lifecycle.dsl.data_retention : undefined
    }), {
      logger
    });

    // if we transition from ilm to dsl or vice versa, the rolled over backing
    // indices need to be updated or they'll retain the lifecycle configuration
    // set at the time of creation.
    // this is not needed for serverless since only dsl is allowed.
    if (isServerless) {
      return;
    }
    const isIlm = (0, _streamsSchema.isIlmLifecycle)(lifecycle);
    await (0, _retry.retryTransientEsErrors)(() =>
    // TODO: use client method once available
    esClient.transport.request({
      method: 'PUT',
      path: `/_data_stream/${names.join(',')}/_settings`,
      body: {
        'index.lifecycle.name': isIlm ? lifecycle.ilm.policy : null,
        'index.lifecycle.prefer_ilm': isIlm
      }
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Error updating data stream lifecycle: ${err.message}`);
    throw err;
  }
}