"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "importTimelines", {
  enumerable: true,
  get: function () {
    return _helpers.importTimelines;
  }
});
exports.importTimelinesRoute = void 0;
var _path = require("path");
var _fp = require("lodash/fp");
var _zodHelpers = require("@kbn/zod-helpers");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../../../../../common/constants");
var _utils = require("../../../../detection_engine/routes/utils");
var _helpers = require("./helpers");
var _timeline = require("../../../../../../common/api/timeline");
var _common = require("../../../utils/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const importTimelinesRoute = (router, config) => {
  router.versioned.post({
    path: `${_constants.TIMELINE_IMPORT_URL}`,
    security: {
      authz: {
        requiredPrivileges: ['timeline_write']
      }
    },
    options: {
      body: {
        maxBytes: config.maxTimelineImportPayloadBytes,
        output: 'stream'
      }
    },
    access: 'public'
  }).addVersion({
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_timeline.ImportTimelinesRequestBody)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    try {
      const siemResponse = (0, _utils.buildSiemResponse)(response);
      const savedObjectsClient = (await context.core).savedObjects.client;
      if (!savedObjectsClient) {
        return siemResponse.error({
          statusCode: 404
        });
      }
      const {
        file,
        isImmutable
      } = request.body;
      const filename = extractFilename(file);
      const fileExtension = (0, _path.extname)(filename).toLowerCase();
      if (fileExtension !== '.ndjson') {
        return siemResponse.error({
          statusCode: 400,
          body: `Invalid file extension ${fileExtension}`
        });
      }
      const frameworkRequest = await (0, _common.buildFrameworkRequest)(context, request);
      const res = await (0, _helpers.importTimelines)(file, config.maxTimelineImportExportSize, frameworkRequest, isImmutable === 'true');
      if (res instanceof Error || typeof res === 'string') {
        throw res;
      } else {
        return response.ok({
          body: res !== null && res !== void 0 ? res : {}
        });
      }
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      const siemResponse = (0, _utils.buildSiemResponse)(response);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.importTimelinesRoute = importTimelinesRoute;
function extractFilename(fileObj) {
  const filename = (0, _fp.get)('hapi.filename', fileObj);
  if (filename && typeof filename === 'string') {
    return filename;
  }
  throw new Error('`filename` missing in file');
}