"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryPrebuiltRuleAlertsTaskConfig = createTelemetryPrebuiltRuleAlertsTaskConfig;
var _constants = require("../constants");
var _helpers = require("../helpers");
var _filterlists = require("../filterlists");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryPrebuiltRuleAlertsTaskConfig(maxTelemetryBatch) {
  const taskName = 'Security Solution - Prebuilt Rule and Elastic ML Alerts Telemetry';
  const taskVersion = '1.2.0';
  const taskType = 'security:telemetry-prebuilt-rule-alerts';
  return {
    type: taskType,
    title: taskName,
    interval: '1h',
    timeout: '15m',
    version: taskVersion,
    getLastExecutionTime: _helpers.getPreviousDailyTaskTimestamp,
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('prebuilt_rule_alerts'), mdc);
      const trace = taskMetricsService.start(taskType);
      log.debug('Running telemetry task');
      try {
        const [clusterInfoPromise, licenseInfoPromise, packageVersion] = await Promise.allSettled([receiver.fetchClusterInfo(), receiver.fetchLicenseInfo(), receiver.fetchDetectionRulesPackageVersion()]);
        const clusterInfo = (0, _helpers.safeValue)(clusterInfoPromise);
        const licenseInfo = (0, _helpers.safeValue)(licenseInfoPromise);
        const packageInfo = (0, _helpers.safeValue)(packageVersion, undefined);
        const index = receiver.getAlertsIndex();
        if (index === undefined) {
          log.warn(`alerts index is not ready yet, skipping telemetry task`);
          await taskMetricsService.end(trace);
          return 0;
        }
        const unflattenedFilterList = (0, _helpers.unflatten)(_filterlists.filterList.prebuiltRulesAlerts);
        for await (const alerts of receiver.fetchPrebuiltRuleAlertsBatch(index, (_taskExecutionPeriod$ = taskExecutionPeriod.last) !== null && _taskExecutionPeriod$ !== void 0 ? _taskExecutionPeriod$ : 'now-1h', taskExecutionPeriod.current)) {
          var _taskExecutionPeriod$;
          if (alerts.length === 0) {
            await taskMetricsService.end(trace);
            return 0;
          }
          const processedAlerts = alerts.map(event => event['kibana.alert.rule.type'] === _constants.DETECTION_RULE_TYPE_ESQL ? (0, _filterlists.copyAllowlistedFields)(unflattenedFilterList, (0, _helpers.unflatten)(event)) : (0, _filterlists.copyAllowlistedFields)(_filterlists.filterList.prebuiltRulesAlerts, event));
          const sanitizedAlerts = processedAlerts.map(event => (0, _helpers.processK8sUsernames)(clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_uuid, event));
          const enrichedAlerts = sanitizedAlerts.map(event => ({
            ...event,
            licence_id: licenseInfo === null || licenseInfo === void 0 ? void 0 : licenseInfo.uid,
            cluster_uuid: clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_uuid,
            cluster_name: clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_name,
            package_version: packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.version,
            task_version: taskVersion
          }));
          log.debug('sending elastic prebuilt alerts', {
            length: enrichedAlerts.length
          });
          const batches = (0, _helpers.batchTelemetryRecords)(enrichedAlerts, maxTelemetryBatch);
          const promises = batches.map(async batch => {
            await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_DETECTION_ALERTS, batch);
          });
          await Promise.all(promises);
        }
        await taskMetricsService.end(trace);
        return 0;
      } catch (error) {
        logger.error('could not complete task', {
          error
        });
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}