"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActionDetailsById = void 0;
var _fetch_action_request_by_id = require("./utils/fetch_action_request_by_id");
var _fetch_action_responses = require("./utils/fetch_action_responses");
var _utils = require("./utils");
var _errors = require("../../../../common/endpoint/errors");
var _errors2 = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get Action Details for a single action id
 * @param esClient
 * @param metadataService
 * @param actionId
 */
const getActionDetailsById = async (esClient, metadataService, actionId) => {
  let normalizedActionRequest;
  let actionResponses;
  try {
    // Get both the Action Request(s) and action Response(s)
    const [actionRequestEsDoc, actionResponseResult] = await Promise.all([
    // Get the action request(s)
    (0, _fetch_action_request_by_id.fetchActionRequestById)(esClient, actionId),
    // Get all responses
    (0, _fetch_action_responses.fetchActionResponses)({
      esClient,
      actionIds: [actionId]
    })]);
    actionResponses = actionResponseResult;
    normalizedActionRequest = (0, _utils.mapToNormalizedActionRequest)(actionRequestEsDoc);
  } catch (error) {
    if ((0, _errors.isEndpointError)(error)) {
      throw error;
    }
    throw new _errors.EndpointError(error.message, error);
  }

  // If action id was not found, error out
  if (!normalizedActionRequest) {
    throw new _errors2.NotFoundError(`Action with id '${actionId}' not found.`);
  }

  // get host metadata info with queried agents
  const agentsHostInfo = normalizedActionRequest.agentType === 'endpoint' ? await (0, _utils.getAgentHostNamesWithIds)({
    esClient,
    metadataService,
    agentIds: normalizedActionRequest.agents
  }) : {};
  return (0, _utils.createActionDetailsRecord)(normalizedActionRequest, actionResponses, agentsHostInfo);
};
exports.getActionDetailsById = getActionDetailsById;