"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkInstallPackages = bulkInstallPackages;
exports.isBulkInstallError = isBulkInstallError;
var _pLimit = _interopRequireDefault(require("p-limit"));
var _lodash = require("lodash");
var _app_context = require("../../app_context");
var Registry = _interopRequireWildcard(require("../registry"));
var _install = require("./install");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function bulkInstallPackages({
  savedObjectsClient,
  packagesToInstall,
  esClient,
  spaceId,
  force,
  prerelease,
  authorizationHeader,
  skipIfInstalled
}) {
  const logger = _app_context.appContextService.getLogger();
  const uniquePackages = (0, _lodash.uniqBy)(packagesToInstall, pkg => {
    if (typeof pkg === 'string') {
      return pkg;
    }
    return pkg.name;
  });
  const limiter = (0, _pLimit.default)(10);
  const packagesResults = await Promise.allSettled(uniquePackages.map(async pkg => {
    return limiter(async () => {
      if (typeof pkg === 'string') {
        return Registry.fetchFindLatestPackageOrThrow(pkg, {
          prerelease
        }).then(pkgRes => ({
          name: pkgRes.name,
          version: pkgRes.version,
          prerelease: undefined,
          skipDataStreamRollover: undefined
        }));
      }
      if (pkg.version !== undefined) {
        return Promise.resolve(pkg);
      }
      return Registry.fetchFindLatestPackageOrThrow(pkg.name, {
        prerelease: prerelease || pkg.prerelease
      }).then(pkgRes => ({
        name: pkgRes.name,
        version: pkgRes.version,
        prerelease: pkg.prerelease,
        skipDataStreamRollover: pkg.skipDataStreamRollover
      }));
    });
  }));
  logger.debug(`kicking off bulk install of ${packagesToInstall.map(pkg => typeof pkg === 'string' ? pkg : pkg.name).join(', ')}`);
  const bulkInstallResults = await Promise.allSettled(packagesResults.map(async (result, index) => {
    const packageName = getNameFromPackagesToInstall(packagesToInstall, index);
    if (result.status === 'rejected') {
      return {
        name: packageName,
        error: result.reason
      };
    }
    const pkgKeyProps = result.value;
    if (!force || skipIfInstalled) {
      const installedPackageResult = await (0, _install.isPackageVersionOrLaterInstalled)({
        savedObjectsClient,
        pkgName: pkgKeyProps.name,
        pkgVersion: pkgKeyProps.version
      });
      if (installedPackageResult) {
        const {
          name,
          version,
          installed_es: installedEs,
          installed_kibana: installedKibana
        } = installedPackageResult.package;
        return {
          name,
          version,
          result: {
            assets: [...installedEs, ...installedKibana],
            status: 'already_installed',
            installType: 'unknown'
          }
        };
      }
    }
    const pkgkey = Registry.pkgToPkgKey(pkgKeyProps);
    const installResult = await (0, _install.installPackage)({
      savedObjectsClient,
      esClient,
      pkgkey,
      installSource: 'registry',
      spaceId,
      force,
      prerelease: prerelease || 'prerelease' in pkgKeyProps && pkgKeyProps.prerelease,
      authorizationHeader,
      skipDataStreamRollover: pkgKeyProps.skipDataStreamRollover
    });
    if (installResult.error) {
      return {
        name: packageName,
        error: installResult.error,
        installType: installResult.installType
      };
    }
    return {
      name: packageName,
      version: pkgKeyProps.version,
      result: installResult
    };
  }));
  return bulkInstallResults.map((result, index) => {
    const packageName = getNameFromPackagesToInstall(packagesToInstall, index);
    if (result.status === 'fulfilled') {
      if (result.value && result.value.error) {
        return {
          name: packageName,
          error: result.value.error,
          installType: result.value.installType
        };
      } else {
        return result.value;
      }
    } else {
      return {
        name: packageName,
        error: result.reason
      };
    }
  });
}
function isBulkInstallError(installResponse) {
  return 'error' in installResponse && installResponse.error instanceof Error;
}
function getNameFromPackagesToInstall(packagesToInstall, index) {
  const entry = packagesToInstall[index];
  if (typeof entry === 'string') return entry;
  return entry.name;
}