"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateMlInferenceMappings = void 0;
var _error_codes = require("../../../../../common/types/error_codes");
var _identify_exceptions = require("../../../../utils/identify_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates Elasticsearch field mappings so that the outputs of ML Inference pipelines are not indexed as `float` fields
 * @param indexName - the index whose mapping will be updated
 * @param fieldMappings - the array of objects representing the source field (text) names and target fields (ML output) names
 * @param esClient
 */
const updateMlInferenceMappings = async (indexName, modelId, fieldMappings, esClient) => {
  // Check if the model is of text_expansion type, if not, skip the mapping update
  if (!(await isTextExpansionModel(modelId, esClient))) {
    return {
      acknowledged: false
    };
  }
  const sourceFields = fieldMappings.map(({
    sourceField
  }) => sourceField);
  const nonDefaultTargetFields = fieldMappings.filter(fieldMapping => fieldMapping.targetField !== `ml.inference.${fieldMapping.sourceField}_expanded`).map(fieldMapping => fieldMapping.targetField);

  // Today, we only update mappings for text_expansion fields.
  const mapping = generateTextExpansionMappingProperties(sourceFields, nonDefaultTargetFields);
  try {
    return await esClient.indices.putMapping({
      index: indexName,
      properties: mapping
    });
  } catch (e) {
    if ((0, _identify_exceptions.isIllegalArgumentException)(e)) {
      throw new Error(_error_codes.ErrorCode.MAPPING_UPDATE_FAILED);
    } else {
      throw e;
    }
  }
};
exports.updateMlInferenceMappings = updateMlInferenceMappings;
const generateTextExpansionMappingProperties = (sourceFields, targetFields) => {
  return {
    ml: {
      properties: {
        inference: {
          properties: {
            ...formDefaultElserMappingProps(sourceFields)
          }
        }
      }
    },
    ...targetFields.reduce((previous, targetField) => ({
      ...previous,
      [targetField]: {
        properties: {
          model_id: {
            type: 'keyword'
          },
          predicted_value: {
            type: 'sparse_vector'
          }
        }
      }
    }), {})
  };
};
const formDefaultElserMappingProps = sourceFields => {
  return sourceFields.reduce((previous, sourceField) => ({
    ...previous,
    [`${sourceField}_expanded`]: {
      properties: {
        model_id: {
          type: 'keyword'
        },
        predicted_value: {
          type: 'sparse_vector'
        }
      }
    }
  }), {});
};
const isTextExpansionModel = async (modelId, esClient) => {
  var _models$trained_model, _models$trained_model2;
  const models = await esClient.ml.getTrainedModels({
    model_id: modelId
  });
  return ((_models$trained_model = models.trained_model_configs[0]) === null || _models$trained_model === void 0 ? void 0 : (_models$trained_model2 = _models$trained_model.inference_config) === null || _models$trained_model2 === void 0 ? void 0 : _models$trained_model2.text_expansion) !== undefined;
};