"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isValidRolloverInterval = exports.getHighestTimeUnit = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns the highest time unit of the given duration
 * (the highest unit with a value higher of equal to 1)
 *
 * @example
 * ```
 * getHighestTimeUnit(moment.duration(4, 'day'))
 * // 'day'
 * getHighestTimeUnit(moment.duration(90, 'minute'))
 * // 'hour' - 90min = 1.5h
 * getHighestTimeUnit(moment.duration(30, 'minute'))
 * // 'minute' - 30min = 0,5h
 * ```
 */
const getHighestTimeUnit = duration => {
  if (duration.asYears() >= 1) {
    return 'year';
  }
  if (duration.asMonths() >= 1) {
    return 'month';
  }
  if (duration.asWeeks() >= 1) {
    return 'week';
  }
  if (duration.asDays() >= 1) {
    return 'day';
  }
  if (duration.asHours() >= 1) {
    return 'hour';
  }
  if (duration.asMinutes() >= 1) {
    return 'minute';
  }
  if (duration.asSeconds() >= 1) {
    return 'second';
  }
  return 'millisecond';
};

/**
 * Returns true if the given duration is valid to be used with by the {@link TimeIntervalTriggeringPolicy | policy}
 *
 * See {@link TimeIntervalTriggeringPolicyConfig.interval} for rules and reasons around this validation.
 */
exports.getHighestTimeUnit = getHighestTimeUnit;
const isValidRolloverInterval = duration => {
  const highestUnit = getHighestTimeUnit(duration);
  const asHighestUnit = duration.as(highestUnit);
  return Number.isInteger(asHighestUnit);
};
exports.isValidRolloverInterval = isValidRolloverInterval;