"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.FullStoryShipper = void 0;
var tslib_1 = require("tslib");
var rxjs_1 = require("rxjs");
var lodash_get_1 = tslib_1.__importDefault(require("lodash.get"));
var lodash_has_1 = tslib_1.__importDefault(require("lodash.has"));
var get_properties_and_schema_1 = require("./get_properties_and_schema");
var load_snippet_1 = require("./load_snippet");
var get_parsed_version_1 = require("./get_parsed_version");
var PAGE_VARS_KEYS = [
    // Page-specific keys
    'pageName',
    'page',
    'entityId',
    'applicationId',
    // Deployment-specific keys
    'version', // x4, split to version_major, version_minor, version_patch for easier filtering
    'buildSha', // Useful for Serverless
    'cloudId',
    'deploymentId',
    'projectId', // projectId and deploymentId are mutually exclusive. They shouldn't be sent in the same offering.
    'cluster_name',
    'cluster_uuid',
    'cluster_version',
    'license_id',
    'license_status',
    'license_type',
    // Session-specific
    'session_id',
    'preferred_languages',
];
// `labels` object keys from page vars.
var PAGE_VARS_LABELS_KEYS = ['serverless'];
/**
 * FullStory shipper.
 */
var FullStoryShipper = /** @class */ (function () {
    /**
     * Creates a new instance of the FullStoryShipper.
     * @param config {@link FullStoryShipperConfig}
     * @param initContext {@link AnalyticsClientInitContext}
     */
    function FullStoryShipper(config, initContext) {
        var _this = this;
        this.initContext = initContext;
        this.pageContext$ = new rxjs_1.Subject();
        this.userContext$ = new rxjs_1.Subject();
        this.subscriptions = new rxjs_1.Subscription();
        var eventTypesAllowlist = config.eventTypesAllowlist, _a = config.pageVarsDebounceTimeMs, pageVarsDebounceTimeMs = _a === void 0 ? 500 : _a, snippetConfig = tslib_1.__rest(config, ["eventTypesAllowlist", "pageVarsDebounceTimeMs"]);
        this.fullStoryApi = (0, load_snippet_1.loadSnippet)(snippetConfig);
        this.eventTypesAllowlist = eventTypesAllowlist;
        this.delayedStartup = snippetConfig.captureOnStartup === false;
        this.subscriptions.add(this.userContext$
            .pipe((0, rxjs_1.distinct)(function (_a) {
            var userId = _a.userId, isElasticCloudUser = _a.isElasticCloudUser, cloudIsElasticStaffOwned = _a.cloudIsElasticStaffOwned, cloudTrialEndDate = _a.cloudTrialEndDate;
            return [userId, isElasticCloudUser, cloudIsElasticStaffOwned, cloudTrialEndDate].join('-');
        }))
            .subscribe(function (userVars) { return _this.updateUserVars(userVars); }));
        this.subscriptions.add(this.pageContext$
            .pipe((0, rxjs_1.map)(function (newContext) {
            // Cherry-picking fields because FS limits the number of fields that can be sent.
            // > Note: You can capture up to 20 unique page properties (exclusive of pageName) for any given page
            // > and up to 500 unique page properties across all pages.
            // https://help.fullstory.com/hc/en-us/articles/1500004101581-FS-setVars-API-Sending-custom-page-data-to-FullStory
            var pageVars = PAGE_VARS_KEYS.reduce(function (acc, key) {
                if ((0, lodash_has_1.default)(newContext, key)) {
                    acc[key] = (0, lodash_get_1.default)(newContext, key);
                }
                return acc;
            }, {});
            var pageLabelsVars = PAGE_VARS_LABELS_KEYS.reduce(function (acc, key) {
                var labelKey = "labels.".concat(key);
                if ((0, lodash_has_1.default)(newContext, labelKey)) {
                    acc.labels = acc.labels || {};
                    acc.labels[key] = (0, lodash_get_1.default)(newContext, labelKey);
                }
                return acc;
            }, {});
            return tslib_1.__assign(tslib_1.__assign({}, pageVars), pageLabelsVars);
        }), (0, rxjs_1.filter)(function (pageVars) { return Object.keys(pageVars).length > 0; }), 
        // Wait for anything to actually change.
        (0, rxjs_1.distinct)(function (pageVars) {
            var sortedKeys = Object.keys(pageVars).sort();
            return sortedKeys.map(function (key) { return pageVars[key]; }).join('-');
        }), 
        // We need some debounce time to ensure everything is updated before calling FS because some properties cannot be changed twice for the same URL.
        (0, rxjs_1.debounceTime)(pageVarsDebounceTimeMs))
            .subscribe(function (pageVars) {
            _this.initContext.logger.debug(function () { return "Calling FS.setProperties/page with context ".concat(JSON.stringify(pageVars)); });
            var _a = (0, get_properties_and_schema_1.getPropertiesAndSchema)(tslib_1.__assign(tslib_1.__assign({}, pageVars), (pageVars.version ? (0, get_parsed_version_1.getParsedVersion)(pageVars.version) : {}))), properties = _a.properties, schema = _a.schema;
            _this.fullStoryApi('setProperties', {
                type: 'page',
                properties: properties,
                schema: schema,
            });
        }));
    }
    /**
     * Calls `fs.identify`, `fs.setUserVars` and `fs.setVars` depending on the fields provided in the newContext.
     * @param newContext The full new context to set {@link EventContext}
     */
    FullStoryShipper.prototype.extendContext = function (newContext) {
        this.initContext.logger.debug(function () { return "Received context ".concat(JSON.stringify(newContext)); });
        // FullStory requires different APIs for different type of contexts:
        // User-level context.
        this.userContext$.next(newContext);
        // Event-level context. At the moment, only the scope `page` is supported by FullStory for webapps.
        this.pageContext$.next(newContext);
    };
    /**
     * Stops/restarts the shipping mechanism based on the value of isOptedIn
     * @param isOptedIn `true` for resume sending events. `false` to stop.
     */
    FullStoryShipper.prototype.optIn = function (isOptedIn) {
        this.initContext.logger.debug("Setting FS to optIn ".concat(isOptedIn));
        // FullStory uses 2 different opt-in methods:
        // - `consent` is needed to allow collecting information about the components
        //   declared as "Record with user consent" (https://help.fullstory.com/hc/en-us/articles/360020623574).
        //   We need to explicitly call `consent` if for the "Record with user content" feature to work.
        this.fullStoryApi('setIdentity', { consent: isOptedIn });
        // - `restart` and `shutdown` fully start/stop the collection of data.
        if (isOptedIn) {
            // When using the delayed startup strategy...
            if (this.delayedStartup) {
                // ... it is recommended to identify the user before enabling capture to avoid duplicated sessions
                // when navigating across multiple domains (Cloud UI <-> Deployment 1 <-> Deployment 2 <-> Project 1).
                // If the user is already present, we can start/resume the capture straight away.
                if (this.lastUserId) {
                    if (this.optedIn === false) {
                        // when previously shut down => restart
                        // https://developer.fullstory.com/browser/auto-capture/capture-data/#restart-data-capture
                        this.fullStoryApi('restart');
                    }
                    else {
                        this.fullStoryApi('start');
                    }
                }
            }
            else {
                this.fullStoryApi('restart');
            }
        }
        else {
            this.fullStoryApi('shutdown');
        }
        // Storing the state because it's necessary to find out whether to start/restart,
        // and it's used to explicitly start after the user is identified in delayedStartup mode.
        this.optedIn = isOptedIn;
    };
    /**
     * Filters the events by the eventTypesAllowlist from the config.
     * Then it transforms the event into a FS valid format and calls `fs.event`.
     * @param events batched events {@link Event}
     */
    FullStoryShipper.prototype.reportEvents = function (events) {
        var _this = this;
        this.initContext.logger.debug("Reporting ".concat(events.length, " events to FS"));
        events
            .filter(function (event) { var _a, _b; return (_b = (_a = _this.eventTypesAllowlist) === null || _a === void 0 ? void 0 : _a.includes(event.event_type)) !== null && _b !== void 0 ? _b : true; })
            .forEach(function (event) {
            // We only read event.properties and discard the rest because the context is already sent in the other APIs.
            var _a = (0, get_properties_and_schema_1.getPropertiesAndSchema)(event.properties), properties = _a.properties, schema = _a.schema;
            _this.fullStoryApi('trackEvent', {
                name: event.event_type,
                properties: properties,
                schema: schema,
            });
        });
    };
    /**
     * Flushes all internal queues of the shipper.
     * It doesn't really do anything inside because this shipper doesn't hold any internal queues.
     */
    FullStoryShipper.prototype.flush = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () { return tslib_1.__generator(this, function (_a) {
            return [2 /*return*/];
        }); });
    };
    /**
     * Shuts down the shipper.
     */
    FullStoryShipper.prototype.shutdown = function () {
        this.subscriptions.unsubscribe();
    };
    FullStoryShipper.prototype.updateUserVars = function (_a) {
        var userId = _a.userId, isElasticCloudUser = _a.isElasticCloudUser, cloudIsElasticStaffOwned = _a.cloudIsElasticStaffOwned, cloudTrialEndDate = _a.cloudTrialEndDate;
        // Call it only when the userId changes
        if (userId && userId !== this.lastUserId) {
            this.initContext.logger.debug("Calling FS.identify with userId ".concat(userId));
            // We need to call the API for every new userId (restarting the session).
            this.fullStoryApi('setIdentity', { uid: userId });
            this.lastUserId = userId;
            // When delayed startup is enabled, we need to manually start capturing after identifying the user.
            if (this.delayedStartup && this.optedIn) {
                this.fullStoryApi('start');
            }
        }
        // User-level context
        if (typeof isElasticCloudUser === 'boolean' || typeof cloudIsElasticStaffOwned === 'boolean' || cloudTrialEndDate) {
            var userVars_1 = {
                isElasticCloudUser: isElasticCloudUser,
                cloudIsElasticStaffOwned: cloudIsElasticStaffOwned,
                cloudTrialEndDate: cloudTrialEndDate,
            };
            this.initContext.logger.debug(function () { return "Calling FS.setProperties/user with ".concat(JSON.stringify(userVars_1)); });
            var _b = (0, get_properties_and_schema_1.getPropertiesAndSchema)(userVars_1), properties = _b.properties, schema = _b.schema;
            this.fullStoryApi('setProperties', {
                type: 'user',
                properties: properties,
                schema: schema,
            });
        }
    };
    /** Shipper's unique name */
    FullStoryShipper.shipperName = 'FullStory';
    return FullStoryShipper;
}());
exports.FullStoryShipper = FullStoryShipper;
