# frozen_string_literal: true

require_dependency File.join(__dir__, 'actastic_record/class_methods')
require_dependency File.join(__dir__, 'actastic_record/errors')

module Actastic
  module ActasticRecord
    extend ActiveSupport::Concern

    RETRY_ON_CONFLICT = 10

    included do
      extend ActiveModel::Naming
      extend ActiveModel::Callbacks

      include ActiveModel::Validations
      include ActiveModel::Validations::Callbacks
      include ActiveModel::Conversion
      include ActiveModel::Serialization

      include Actastic::Dirty
      include Actastic::Validations
      include Actastic::AssociationCache
      include Actastic::Serialization

      extend OrmAdapter::ToAdapter
      self::OrmAdapter = OrmAdapter::ActasticRecord

      define_model_callbacks :save, :create, :update, :destroy, :initialize

      class_attribute :schema, :current_scope, :instance_reader => false, :instance_predicate => false
      cattr_accessor :inheritance_column, :association_definitions, :instance_accessor => false

      self.inheritance_column = :type_
      self.association_definitions = {}
      self.current_scope = Concurrent::ThreadLocalVar.new

      extend Devise::Models
    end

    delegate :schema, :logger, :instrument,
             :field?, :encrypted_field?, :id_field?, :immutable_field?,
             :find, :cast, :to => :class

    attr_accessor :version

    def initialize(attributes = {})
      run_callbacks(:initialize) do
        @version = DocumentVersion.new_from_attributtes(attributes)
        @attributes = {}
        assign_attributes_from_user(attributes.except(*DocumentVersion::ATTRIBUTES))
      end
    end

    def initialize_from_db(attributes)
      run_callbacks(:initialize) do
        @version = DocumentVersion.new_from_attributtes(attributes)
        @attributes = {}
        assign_attributes_from_db(attributes.except(*DocumentVersion::ATTRIBUTES))
        clear_changes_information
      end
    end

    def ==(other)
      other.instance_of?(self.class) && !id.nil? && id == other.id
    end
    alias_method :eql?, :==

    def reload
      doc = find(id)
      @version = doc.version
      assign_attributes_from_db(doc.attributes.reject { |name| id_field?(name) })
      clear_changes_information
      self
    end

    def new_record?
      @version.new_record?
    end

    def destroyed?
      !!@destroyed
    end

    def persisted?
      !new_record? && !destroyed?
    end

    def save(options={})
      validate = options.has_key?(:validate) ? options[:validate] : true
      version = options.has_key?(:version) ? options[:version] : false
      context = options.has_key?(:context) ? options[:context] : nil

      set_type_if_missing!

      create_or_update_callback = new_record? ? :create : :update
      return false if validate && !valid?(context)

      run_callbacks(:save) do
        run_callbacks(create_or_update_callback) do
          save_without_callbacks!(:version => version)
        end
      end

      changes_applied

      true
    end

    def save!(**options)
      save(options) || raise(ActasticRecord::RecordInvalid.new(self))
    end

    def update(attributes)
      assign_attributes_from_user(attributes)
      save
    end
    alias_method :update_attributes, :update

    def update!(attributes)
      assign_attributes_from_user(attributes)
      save!
    end
    alias_method :update_attributes!, :update!

    def update_attribute(field, value)
      assign_attributes_from_user(field => value)
      save({:validate => false})
    end

    def set(attributes)
      assign_attributes_from_user(attributes)
      save_without_callbacks!
    end

    def delete
      raise 'cannot destroy new records' if new_record?
      instrument('delete', :id => id) do
        schema.index.remove(id, :refresh => true)
      end
      delete_unique_field_constraints!
      @destroyed = true
    end

    def destroy
      run_callbacks(:destroy) { delete }
    end

    def destroy!
      destroy || raise(ActasticRecord::RecordNotDestroyed.new('Failed to destroy the record', self))
    end

    def touch
      raise ActasticRecordError, 'cannot touch an object that is not persisted' unless persisted?

      right_now = Time.zone.now
      self.updated_at = right_now

      clear_attribute_changes(:updated_at)

      begin
        find(id).update(:updated_at => right_now)
      rescue ActasticRecord::RecordNotFound
        false
      end
    end

    def increment(attribute, by = 1)
      self[attribute] ||= 0
      self[attribute] += by
      self
    end

    def increment!(attribute, by = 1)
      increment(attribute, by).update_attribute(attribute, self[attribute])
    end

    def atomic_array_add!(attribute, values)
      instrument('atomic_add', :id => id, :attribute => attribute) do
        atomic_array_operation!(attribute, values, <<~SCRIPT.strip, &:|)
          ctx._source[params.attribute].removeAll(params.values);
          ctx._source[params.attribute].addAll(params.values);
        SCRIPT
      end
    end

    def atomic_array_remove!(attribute, values)
      instrument('atomic_remove', :id => id, :attribute => attribute) do
        atomic_array_operation!(attribute, values, <<~SCRIPT.strip, &:-)
          ctx._source[params.attribute].removeAll(params.values);
        SCRIPT
      end
    end

    def [](attribute)
      attribute = attribute.to_sym
      return unless field?(attribute)
      fetch_attribute(attribute)
    end

    def []=(attribute, value)
      attribute = attribute.to_sym
      unless field?(attribute)
        raise ActiveModel::MissingAttributeError, "can't write unknown attribute `#{attribute}`"
      end
      assign_attribute(attribute, value)
    end

    def assign_attributes(attributes)
      assign_attributes_from_user(attributes)
    end
    alias_method :attributes=, :assign_attributes

    def attributes
      schema.fields.each_key.each_with_object({}) do |name, out|
        out[name] = fetch_attribute(name)
      end
    end

    def as_json(_options)
      attributes
    end

    def slice(*attributes)
      attributes.each_with_object({}) { |name, out| out[name] = public_send(name) }
    end

    delegate :hash, :to => :attributes

    private

    attr_accessor :scope_filter, :exclude_scope_filter

    def attribute_might_change!(attribute, value)
      return if encrypted_field?(attribute)
      public_send("#{attribute}_will_change!") if value.is_a?(Hash) || value.is_a?(Array)
    end

    def fetch_attribute(attribute)
      # return the attribute if it's available
      if @attributes.key?(attribute)
        value = @attributes.fetch(attribute)
        attribute_might_change!(attribute, value)
        return value
      end

      # return if there is no default value
      field = schema.fields.fetch(attribute)
      default = field[:default]
      return if default.nil?

      # set the default value and return it
      value = cast(default.deep_dup, field.fetch(:type), field)
      attribute_might_change!(attribute, value)
      @attributes[attribute] = value
    end

    def assign_attribute(attribute, value)
      if immutable_field?(attribute) && !new_record? && @attributes.key?(attribute)
        raise InvalidUpdate, "#{attribute} cannot be updated"
      end

      field = schema.fields.fetch(attribute)

      existing_attribute_has_new_value = @attributes.key?(attribute) && !attribute_equals?(attribute, value)
      unassigned_attribute_is_not_default = !@attributes.key?(attribute) && field[:default] != value

      if existing_attribute_has_new_value || unassigned_attribute_is_not_default
        public_send("#{attribute}_will_change!")
      end

      type = (attribute == :id && schema.id == :id) ? :bson_id : field.fetch(:type)
      @attributes[attribute] = cast(value, type, field)
    end

    def assign_attributes_from_user(attributes)
      attributes.each { |attribute, value| public_send("#{attribute}=", value) }
    end

    def assign_attributes_from_db(attributes)
      attributes.each { |attribute, value| assign_attribute(attribute, value) }
    end

    # This behavior is copied in Migration.format_document_id. If you change this, change the other as well.
    def set_id_if_missing!
      self.id ||=
        case schema.id
        when :id
          BSON::ObjectId.new
        when Array
          schema.id.map do |field|
            value = public_send(field)
            raise InvalidIdValue, field if value.blank?
            "#{field}:#{value}"
          end.join('|')
        else
          public_send(schema.id)
        end
    end

    def set_type_if_missing!
      inheritance_column = self.class.inheritance_column
      return unless field?(inheritance_column)
      unless public_send(inheritance_column)
        public_send("#{inheritance_column}=", self.class.name)
      end
    end

    def save_options(version)
      { :refresh => true }.tap do |options|
        if new_record?
          options[:op_type] = 'create'
        elsif version
          options.merge!(@version.save_options)
        else
          options[:retry_on_conflict] = RETRY_ON_CONFLICT
        end
      end
    end

    def save_without_callbacks!(version: false)
      return unless new_record? || changed?

      set_id_if_missing!
      self.updated_at = Time.zone.now unless updated_at_changed?
      save_unique_field_constraints!

      data =
        if new_record?
          self.created_at ||= updated_at
          instrument('create', :id => id) do
            schema.index.store(id, to_json, **save_options(version))
          end
        else
          begin
            instrument('update', :id => id) do
              schema.index.update(id, changes_json, **save_options(version))
            end
          rescue Swiftype::ES::VersionConflictEngineException
            raise ActasticRecord::StaleObjectError
          end
        end
      unless data['result'] == 'noop'
        @version = DocumentVersion.new_from_es_doc(data)
      end
      true
    end

    def save_unique_field_constraints!
      rollback_procs = []
      options = { :op_type => 'create' }

      schema.unique_constraint_indices_by_field_names.each do |field_names, index|
        next unless field_names.any? { |field_name| attribute_changed?(field_name.to_s) }

        value = field_names.map do |field_name|
          if encrypted_field?(field_name)
            @attributes.dig(field_name, :hash)
          else
            @attributes[field_name]
          end
        end.join('|')

        begin
          index.store(value, { schema.foreign_key => id }.to_json, **options)
        rescue Swiftype::ES::VersionConflictEngineException
          raise ActasticRecord::RecordNotUnique, "#{value.inspect} not unique value for field #{field_names.join(', ')}"
        end

        rollback_procs << lambda do
          logger.warn { "Rolling back secondary index write #{value.inspect} for field #{field_names.join(', ')}" }
          index.remove(value)
        end
      end
    rescue StandardError => e
      if rollback_procs.present?
        logger.warn { "Caught #{e.class}: #{e.message}; attempting to roll back #{rollback_procs.size} secondary writes" }
        rollback_procs.reverse_each(&:call)
      end
      raise
    end

    def attribute_equals?(attribute, value)
      attribute_value = fetch_attribute(attribute)
      if attribute_value.is_a?(Array) && value.is_a?(Array) && !attribute_value.first.is_a?(Hash) && !value.first.is_a?(Hash)
        attribute_value.sort == value.sort
      elsif encrypted_field?(attribute)
        attribute_value&.symbolize_keys&.fetch(:hash) == value&.symbolize_keys&.fetch(:hash)
      else
        attribute_value == value
      end
    end

    def delete_unique_field_constraints!
      schema.unique_constraint_indices_by_field_names.each_value.reverse_each do |index_struct|
        index_struct.refresh

        # have to assign values to variables to be used in the delete_by_query block since it changes scope
        foreign_key = schema.foreign_key
        value = id
        index_struct.es_index.delete_by_query do
          term foreign_key, value
        end
      end
    end

    def atomic_array_operation!(attribute, values, script)
      assign_attributes_from_user(attribute => yield(public_send(attribute), values))
      raise ActasticRecord::RecordInvalid.new(self) unless valid?
      if persisted?
        response = self.class.script_update(
          id,
          script,
          :source => attribute,
          :attribute => attribute,
          :values => values
        )
        assign_attributes_from_user(response.fetch(:get).fetch(:_source))
      end
    end

    Actastic::Stats.measure_methods(
      self,
      'actastic.actastic_record',
      :save_without_callbacks!,
      :save_unique_field_constraints!,
      :delete
    )
  end
end
