"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.accessKnownApmEventFields = accessKnownApmEventFields;
var _unflatten_known_fields = require("./unflatten_known_fields");
var _utility_types = require("./utility_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * WeakMap storing all proxied methods for `ApmDocument`. As long as the proxy
 * object lives, the methods will be available. All the methods here are lazily
 * evaluated/initialised so the user only "pays the cost" of the methods they use.
 */
const PROXIED_METHODS = new WeakMap();

/**
 * A Proxied APM document that is strongly typed and runtime checked to be correct.
 * Accessing fields from the document will correctly return single or multi values
 * according to known field types.
 */

/**
 * Interface for exposing the `unflatten()` and `requireFields()` methods on proxied APM documents.
 */

/**
 * An APM document that is strongly typed and runtime checked to be correct.
 * Accessing fields from the document will correctly return single or multi values
 * according to known field types.
 *
 * An `unflatten()` method is also exposed by this document to return an unflattened
 * version of the document.
 */

/**
 * Validates an APM Event document, checking if it has all the required fields if provided,
 * returning a proxied version of the document to allow strongly typed access to known single
 * or multi-value fields. The proxy also exposes an `unflatten()` method to return an unflattened
 * version of the document.
 *
 * ## Example
 *
 * ```ts
 * const event = accessKnownApmEventFields(hit).requireFields(['service.name']);
 *
 * // The key is strongly typed to be `keyof FlattenedApmEvent`.
 * console.log(event['service.name']) // => outputs `"node-svc"`, not `["node-svc"]` as in the original doc
 *
 * const unflattened = event.unflatten();
 *
 * console.log(unflattened.service.name); // => outputs "node-svc" like above
 */

function accessKnownApmEventFields(fields) {
  const proxy = new Proxy(fields, accessHandler);

  // Methods are lazily initialised so you only pay the "cost" of what you use.
  PROXIED_METHODS.set(proxy, {});
  return proxy;
}
const accessHandler = {
  get(fields, key, proxy) {
    switch (key) {
      case 'unflatten':
        // Lazily initialise method on first access
        return PROXIED_METHODS.get(proxy).unflatten ??= () => (0, _unflatten_known_fields.unflattenKnownApmEventFields)(fields);
      case 'requireFields':
        // Lazily initialise method on first access
        return PROXIED_METHODS.get(proxy).requireFields ??= requiredFields => {
          (0, _utility_types.ensureRequiredApmFields)(fields, requiredFields);
          return proxy;
        };
      default:
        {
          const value = fields[key];
          return _utility_types.KNOWN_SINGLE_VALUED_FIELDS_SET.has(key) && Array.isArray(value) ? value[0] : value;
        }
    }
  },
  // Trap any setters to make the proxied object immutable.
  set() {
    return false;
  }
};