/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.FailedNodeException;
import org.elasticsearch.action.admin.cluster.node.info.NodesInfoRequest;
import org.elasticsearch.action.admin.cluster.node.info.NodesInfoResponse;
import org.elasticsearch.client.internal.ClusterAdminClient;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.cluster.service.MasterServiceTaskQueue;
import org.elasticsearch.cluster.version.CompatibilityVersions;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.features.FeatureService;
import org.elasticsearch.features.NodeFeature;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class CompatibilityVersionsFixupListener
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(CompatibilityVersionsFixupListener.class);
    static final NodeFeature FIX_TRANSPORT_VERSION = new NodeFeature("transport.fix_transport_version");
    private static final TimeValue RETRY_TIME = TimeValue.timeValueSeconds((long)30L);
    private final MasterServiceTaskQueue<NodeCompatibilityVersionsTask> taskQueue;
    private final ClusterAdminClient client;
    private final Scheduler scheduler;
    private final Executor executor;
    private final Set<String> pendingNodes = Collections.synchronizedSet(new HashSet());
    private final FeatureService featureService;

    public CompatibilityVersionsFixupListener(ClusterService service, ClusterAdminClient client, FeatureService featureService, ThreadPool threadPool) {
        this(service.createTaskQueue("fixup-transport-versions", Priority.LOW, new TransportVersionUpdater()), client, featureService, threadPool, threadPool.executor("cluster_coordination"));
    }

    CompatibilityVersionsFixupListener(MasterServiceTaskQueue<NodeCompatibilityVersionsTask> taskQueue, ClusterAdminClient client, FeatureService featureService, Scheduler scheduler, Executor executor) {
        this.taskQueue = taskQueue;
        this.client = client;
        this.featureService = featureService;
        this.scheduler = scheduler;
        this.executor = executor;
    }

    @SuppressForbidden(reason="maintaining ClusterState#compatibilityVersions requires reading them")
    private static Map<String, CompatibilityVersions> getCompatibilityVersions(ClusterState clusterState) {
        return clusterState.compatibilityVersions();
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.localNodeMaster()) {
            return;
        }
        ArrayList<Stream<String>> queries = new ArrayList<Stream<String>>();
        Map<String, CompatibilityVersions> compatibilityVersions = CompatibilityVersionsFixupListener.getCompatibilityVersions(event.state());
        if (this.featureService.clusterHasFeature(event.state(), FIX_TRANSPORT_VERSION) && event.state().getMinTransportVersion().equals(ClusterState.INFERRED_TRANSPORT_VERSION)) {
            queries.add(compatibilityVersions.entrySet().stream().filter(e -> ((CompatibilityVersions)e.getValue()).transportVersion().equals(ClusterState.INFERRED_TRANSPORT_VERSION)).map(Map.Entry::getKey));
        }
        queries.add(event.state().nodes().stream().filter(n -> n.getVersion().after(Version.V_8_16_0)).map(DiscoveryNode::getId).filter(n -> compatibilityVersions.getOrDefault(n, CompatibilityVersions.EMPTY).systemIndexMappingsVersion().isEmpty()));
        Set<String> queryNodes = queries.stream().flatMap(Function.identity()).collect(Collectors.toSet());
        if (!queryNodes.isEmpty()) {
            logger.debug("Fetching actual compatibility versions for nodes {}", new Object[]{queryNodes});
            this.updateTransportVersions(queryNodes, 0);
        }
    }

    private void scheduleRetry(Set<String> nodes, int thisRetryNum) {
        logger.debug("Scheduling retry {} for nodes {}", new Object[]{thisRetryNum + 1, nodes});
        this.scheduler.schedule(() -> this.updateTransportVersions(nodes, thisRetryNum + 1), RETRY_TIME, this.executor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateTransportVersions(Set<String> nodes, final int retryNum) {
        final HashSet<String> outstandingNodes = Sets.newHashSetWithExpectedSize(nodes.size());
        Set<String> set = this.pendingNodes;
        synchronized (set) {
            for (String n : nodes) {
                if (!this.pendingNodes.add(n)) continue;
                outstandingNodes.add(n);
            }
        }
        if (outstandingNodes.isEmpty()) {
            return;
        }
        NodesInfoRequest request = new NodesInfoRequest((String[])outstandingNodes.toArray(String[]::new));
        request.clear();
        this.client.nodesInfo(request, new ActionListener<NodesInfoResponse>(){

            @Override
            public void onResponse(NodesInfoResponse response) {
                CompatibilityVersionsFixupListener.this.pendingNodes.removeAll(outstandingNodes);
                CompatibilityVersionsFixupListener.this.handleResponse(response, retryNum);
            }

            @Override
            public void onFailure(Exception e) {
                CompatibilityVersionsFixupListener.this.pendingNodes.removeAll(outstandingNodes);
                logger.warn("Could not read nodes info for nodes {}", new Object[]{outstandingNodes, e});
                CompatibilityVersionsFixupListener.this.scheduleRetry(outstandingNodes, retryNum);
            }
        });
    }

    private void handleResponse(NodesInfoResponse response, int retryNum) {
        Map<String, CompatibilityVersions> results;
        if (response.hasFailures()) {
            HashSet<String> failedNodes = new HashSet<String>();
            for (FailedNodeException fne : response.failures()) {
                logger.warn("Failed to read transport version info from node {}", new Object[]{fne.nodeId(), fne});
                failedNodes.add(fne.nodeId());
            }
            this.scheduleRetry(failedNodes, retryNum);
        }
        if (!(results = response.getNodes().stream().collect(Collectors.toUnmodifiableMap(n -> n.getNode().getId(), n -> new CompatibilityVersions(n.getTransportVersion(), n.getCompatibilityVersions())))).isEmpty()) {
            this.taskQueue.submitTask("update-transport-version", new NodeCompatibilityVersionsTask(results, retryNum), null);
        }
    }

    static class TransportVersionUpdater
    implements ClusterStateTaskExecutor<NodeCompatibilityVersionsTask> {
        TransportVersionUpdater() {
        }

        @Override
        public ClusterState execute(ClusterStateTaskExecutor.BatchExecutionContext<NodeCompatibilityVersionsTask> context) throws Exception {
            ClusterState.Builder builder = ClusterState.builder(context.initialState());
            boolean modified = false;
            for (ClusterStateTaskExecutor.TaskContext<NodeCompatibilityVersionsTask> c : context.taskContexts()) {
                for (Map.Entry<String, CompatibilityVersions> e : c.getTask().results().entrySet()) {
                    Map<String, CompatibilityVersions> cvMap = builder.compatibilityVersions();
                    CompatibilityVersions currentCompatibilityVersions = cvMap.get(e.getKey());
                    TransportVersion recordedTv = Optional.ofNullable(currentCompatibilityVersions).map(CompatibilityVersions::transportVersion).orElse(null);
                    assert (currentCompatibilityVersions != null || !context.initialState().nodes().nodeExists(e.getKey())) : "Node " + e.getKey() + " is in the cluster but does not have an associated transport version recorded";
                    Map systemIndexMappingsVersion = Optional.ofNullable(currentCompatibilityVersions).map(CompatibilityVersions::systemIndexMappingsVersion).orElse(Map.of());
                    if (!Objects.equals(recordedTv, ClusterState.INFERRED_TRANSPORT_VERSION) && !systemIndexMappingsVersion.isEmpty()) continue;
                    builder.putCompatibilityVersions(e.getKey(), e.getValue().transportVersion(), e.getValue().systemIndexMappingsVersion());
                    modified = true;
                }
                c.success(() -> {});
            }
            return modified ? builder.build() : context.initialState();
        }
    }

    class NodeCompatibilityVersionsTask
    implements ClusterStateTaskListener {
        private final Map<String, CompatibilityVersions> results;
        private final int retryNum;

        NodeCompatibilityVersionsTask(Map<String, CompatibilityVersions> results, int retryNum) {
            this.results = results;
            this.retryNum = retryNum;
        }

        @Override
        public void onFailure(Exception e) {
            logger.error("Could not apply compatibility versions for nodes {} to cluster state", new Object[]{this.results.keySet(), e});
            CompatibilityVersionsFixupListener.this.scheduleRetry(this.results.keySet(), this.retryNum);
        }

        public Map<String, CompatibilityVersions> results() {
            return this.results;
        }
    }
}

