/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.logstash.filters.elasticintegration;

import co.elastic.logstash.api.Event;
import co.elastic.logstash.api.FilterMatchListener;
import co.elastic.logstash.filters.elasticintegration.EventProcessorBuilder;
import co.elastic.logstash.filters.elasticintegration.EventToIndexNameResolver;
import co.elastic.logstash.filters.elasticintegration.EventToPipelineNameResolver;
import co.elastic.logstash.filters.elasticintegration.IndexNameToPipelineNameResolver;
import co.elastic.logstash.filters.elasticintegration.IngestDuplexMarshaller;
import co.elastic.logstash.filters.elasticintegration.IngestPipeline;
import co.elastic.logstash.filters.elasticintegration.IngestPipelineResolver;
import co.elastic.logstash.filters.elasticintegration.IntegrationBatch;
import co.elastic.logstash.filters.elasticintegration.IntegrationRequest;
import co.elastic.logstash.filters.elasticintegration.resolver.Resolver;
import co.elastic.logstash.filters.elasticintegration.util.EventUtil;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import java.io.Closeable;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.ingest.IngestDocument;
import org.elasticsearch.ingest.LogstashInternalBridge;
import org.elasticsearch.ingest.common.FailProcessorException;

public class EventProcessor
implements Closeable {
    private final FilterMatchListener filterMatchListener;
    private final IngestPipelineResolver internalPipelineProvider;
    private final EventToPipelineNameResolver eventToPipelineNameResolver;
    private final EventToIndexNameResolver eventToIndexNameResolver;
    private final IndexNameToPipelineNameResolver indexNameToPipelineNameResolver;
    private final IngestDuplexMarshaller eventMarshaller;
    private final List<Closeable> resourcesToClose;
    private static final Logger LOGGER = LogManager.getLogger(EventProcessor.class);
    private static final String METADATA_FAILURE_TEMPLATE = "[@metadata][_ingest_pipeline_failure][%s]";
    private static final String TARGET_PIPELINE_FIELD = "[@metadata][target_ingest_pipeline]";
    static final String PIPELINE_MAGIC_NONE = "_none";

    EventProcessor(FilterMatchListener filterMatchListener, IngestPipelineResolver internalPipelineProvider, EventToPipelineNameResolver eventToPipelineNameResolver, EventToIndexNameResolver eventToIndexNameResolver, IndexNameToPipelineNameResolver indexNameToPipelineNameResolver, Collection<Closeable> resourcesToClose) {
        this.filterMatchListener = filterMatchListener;
        this.internalPipelineProvider = internalPipelineProvider;
        this.eventToIndexNameResolver = eventToIndexNameResolver;
        this.eventToPipelineNameResolver = eventToPipelineNameResolver;
        this.indexNameToPipelineNameResolver = indexNameToPipelineNameResolver;
        this.resourcesToClose = List.copyOf(resourcesToClose);
        this.eventMarshaller = IngestDuplexMarshaller.defaultInstance();
    }

    public static EventProcessorBuilder builder() {
        return new EventProcessorBuilder();
    }

    private static void throwingHandler(Exception e) {
        throw new RuntimeException(e);
    }

    public Collection<Event> processEvents(Collection<Event> incomingEvents) throws InterruptedException, TimeoutException {
        CountDownLatch latch = new CountDownLatch(1);
        IntegrationBatch batch = new IntegrationBatch(incomingEvents);
        try (RefCountingRunnable ref = new RefCountingRunnable(latch::countDown);){
            batch.eachRequest(ref::acquire, this::processRequest);
        }
        if (!latch.await(300L, TimeUnit.SECONDS)) {
            throw new TimeoutException("breaker: catastrophic batch limit reached");
        }
        return batch.events;
    }

    void processRequest(IntegrationRequest request) {
        try {
            Optional resolvedIndexName = this.eventToIndexNameResolver.resolve(request.event(), EventProcessor::throwingHandler);
            Optional<Object> resolvedPipelineName = Objects.nonNull(this.eventToPipelineNameResolver) ? EventProcessor.resolve(request.event(), this.eventToPipelineNameResolver) : (resolvedIndexName.isPresent() ? EventProcessor.resolve((String)resolvedIndexName.get(), this.indexNameToPipelineNameResolver) : Optional.empty());
            if (resolvedPipelineName.isEmpty()) {
                LOGGER.debug(() -> String.format("No pipeline resolved for event %s", EventUtil.serializeEventForLog(LOGGER, request.event())));
                request.complete();
                return;
            }
            String pipelineName = (String)resolvedPipelineName.get();
            if (pipelineName.equals(PIPELINE_MAGIC_NONE)) {
                LOGGER.debug(() -> String.format("Ingest Pipeline bypassed with pipeline `%s` for event `%s`", pipelineName, EventUtil.serializeEventForLog(LOGGER, request.event())));
                request.complete();
                return;
            }
            Optional<IngestPipeline> loadedPipeline = EventProcessor.resolve(pipelineName, this.internalPipelineProvider);
            if (loadedPipeline.isEmpty()) {
                LOGGER.warn(() -> String.format("Pipeline `%s` could not be loaded", pipelineName));
                request.complete(incomingEvent -> EventProcessor.annotateIngestPipelineFailure(incomingEvent, pipelineName, Map.of("message", "pipeline not loaded")));
                return;
            }
            IngestPipeline ingestPipeline = loadedPipeline.get();
            LOGGER.trace(() -> String.format("Using loaded pipeline `%s` (%s)", pipelineName, System.identityHashCode(ingestPipeline)));
            IngestDocument ingestDocument = this.eventMarshaller.toIngestDocument(request.event());
            resolvedIndexName.ifPresent(indexName -> {
                ingestDocument.getMetadata().setIndex((String)indexName);
                ingestDocument.updateIndexHistory((String)indexName);
            });
            this.executePipeline(ingestDocument, ingestPipeline, request);
        }
        catch (Exception e) {
            LOGGER.error(() -> String.format("exception processing event: %s", e.getMessage()));
            request.complete(incomingEvent -> EventProcessor.annotateIngestPipelineFailure(incomingEvent, "UNKNOWN", Map.of("message", e.getMessage(), "exception", e.getClass().getName())));
        }
    }

    private void executePipeline(IngestDocument ingestDocument, IngestPipeline ingestPipeline, IntegrationRequest request) {
        String pipelineName = ingestPipeline.getId();
        String originalIndex = ingestDocument.getMetadata().getIndex();
        ingestPipeline.execute(ingestDocument, (resultIngestDocument, ingestPipelineException) -> {
            if (Objects.nonNull(ingestPipelineException)) {
                Throwable unwrappedException = EventProcessor.unwrapException(ingestPipelineException);
                LOGGER.warn(() -> String.format("ingest pipeline `%s` failed", pipelineName), unwrappedException);
                request.complete(incomingEvent -> EventProcessor.annotateIngestPipelineFailure(incomingEvent, pipelineName, Map.of("message", unwrappedException.getMessage(), "exception", unwrappedException.getClass().getName())));
            } else if (Objects.isNull(resultIngestDocument)) {
                request.complete(incomingEvent -> {
                    LOGGER.trace(() -> String.format("event cancelled by ingest pipeline `%s`: %s", pipelineName, EventUtil.serializeEventForLog(LOGGER, incomingEvent)));
                    incomingEvent.cancel();
                });
            } else {
                String newIndex = resultIngestDocument.getMetadata().getIndex();
                if (!Objects.equals(originalIndex, newIndex) && LogstashInternalBridge.isReroute(resultIngestDocument)) {
                    boolean cycle;
                    LogstashInternalBridge.resetReroute(resultIngestDocument);
                    boolean bl = cycle = !resultIngestDocument.updateIndexHistory(newIndex);
                    if (cycle) {
                        request.complete(incomingEvent -> EventProcessor.annotateIngestPipelineFailure(incomingEvent, pipelineName, Map.of("message", Strings.format("index cycle detected while processing pipeline [%s]: %s + %s", pipelineName, resultIngestDocument.getIndexHistory(), newIndex))));
                        return;
                    }
                    Optional<String> reroutePipelineName = EventProcessor.resolve(newIndex, this.indexNameToPipelineNameResolver);
                    if (reroutePipelineName.isPresent() && !reroutePipelineName.get().equals(PIPELINE_MAGIC_NONE)) {
                        Optional<IngestPipeline> reroutePipeline = EventProcessor.resolve(reroutePipelineName.get(), this.internalPipelineProvider);
                        if (reroutePipeline.isEmpty()) {
                            request.complete(incomingEvent -> EventProcessor.annotateIngestPipelineFailure(incomingEvent, pipelineName, Map.of("message", Strings.format("reroute failed to load next pipeline [%s]: %s -> %s", pipelineName, resultIngestDocument.getIndexHistory(), reroutePipelineName.get()))));
                        } else {
                            this.executePipeline((IngestDocument)resultIngestDocument, reroutePipeline.get(), request);
                        }
                        return;
                    }
                }
                request.complete(incomingEvent -> {
                    Event resultEvent = this.eventMarshaller.toLogstashEvent((IngestDocument)resultIngestDocument);
                    resultEvent.setField(TARGET_PIPELINE_FIELD, (Object)PIPELINE_MAGIC_NONE);
                    this.filterMatchListener.filterMatched(resultEvent);
                    LOGGER.trace(() -> String.format("event transformed by ingest pipeline `%s`%s", pipelineName, EventProcessor.diff(incomingEvent, resultEvent)));
                    incomingEvent.cancel();
                    return resultEvent;
                });
            }
        });
    }

    private static void annotateIngestPipelineFailure(Event event, String pipelineName, Map<String, String> meta) {
        event.tag("_ingest_pipeline_failure");
        event.setField(String.format(METADATA_FAILURE_TEMPLATE, "pipeline"), (Object)pipelineName);
        meta.forEach((metaKey, metaValue) -> event.setField(String.format(METADATA_FAILURE_TEMPLATE, metaKey), metaValue));
    }

    private static Throwable unwrapException(Exception exception) {
        if (exception.getCause() instanceof FailProcessorException) {
            return exception.getCause();
        }
        return exception;
    }

    private static String diff(Event original, Event changed) {
        if (LOGGER.isTraceEnabled()) {
            Map<String, Object> flatOriginal = org.elasticsearch.common.util.Maps.flatten(EventUtil.eventAsMap(original), true, false);
            Map<String, Object> flatChanged = org.elasticsearch.common.util.Maps.flatten(EventUtil.eventAsMap(changed), true, false);
            MapDifference difference = Maps.difference(flatOriginal, flatChanged);
            return String.format(": REMOVING{%s} CHANGING{%s} ADDING{%s}", difference.entriesOnlyOnLeft(), difference.entriesDiffering(), difference.entriesOnlyOnRight());
        }
        return "";
    }

    private static <T, R> Optional<R> resolve(T resolvable, Resolver<T, R> resolver) {
        return resolver.resolve(resolvable, EventProcessor::throwingHandler);
    }

    @Override
    public void close() throws IOException {
        IOUtils.closeWhileHandlingException(this.resourcesToClose);
    }
}

