/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.support;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.TransportActions;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.NotMasterException;
import org.elasticsearch.cluster.SimpleBatchedExecutor;
import org.elasticsearch.cluster.coordination.FailedToCommitClusterStateException;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.cluster.service.MasterServiceTaskQueue;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.component.LifecycleListener;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.features.FeatureService;
import org.elasticsearch.features.NodeFeature;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.engine.DocumentMissingException;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.indices.IndexClosedException;
import org.elasticsearch.indices.IndexPrimaryShardNotAllocatedException;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.security.action.role.BulkRolesResponse;
import org.elasticsearch.xpack.core.security.authz.RoleDescriptor;
import org.elasticsearch.xpack.core.security.authz.store.ReservedRolesStore;
import org.elasticsearch.xpack.security.authz.store.FileRolesStore;
import org.elasticsearch.xpack.security.authz.store.NativeRolesStore;
import org.elasticsearch.xpack.security.support.QueryableBuiltInRoles;
import org.elasticsearch.xpack.security.support.QueryableBuiltInRolesProviderFactory;
import org.elasticsearch.xpack.security.support.QueryableBuiltInRolesUtils;
import org.elasticsearch.xpack.security.support.SecurityIndexManager;

public final class QueryableBuiltInRolesSynchronizer
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(QueryableBuiltInRolesSynchronizer.class);
    public static final boolean QUERYABLE_BUILT_IN_ROLES_ENABLED;
    public static final NodeFeature QUERYABLE_BUILT_IN_ROLES_FEATURE;
    public static final String METADATA_QUERYABLE_BUILT_IN_ROLES_DIGEST_KEY = "queryable_built_in_roles_digest";
    private static final SimpleBatchedExecutor<MarkRolesAsSyncedTask, Map<String, String>> MARK_ROLES_AS_SYNCED_TASK_EXECUTOR;
    private final MasterServiceTaskQueue<MarkRolesAsSyncedTask> markRolesAsSyncedTaskQueue;
    private final ClusterService clusterService;
    private final FeatureService featureService;
    private final QueryableBuiltInRoles.Provider rolesProvider;
    private final NativeRolesStore nativeRolesStore;
    private final Executor executor;
    private final AtomicBoolean synchronizationInProgress = new AtomicBoolean(false);
    private volatile boolean securityIndexDeleted = false;
    static final int MAX_FAILED_SYNC_ATTEMPTS = 10;
    private final AtomicInteger failedSyncAttempts = new AtomicInteger(0);

    public QueryableBuiltInRolesSynchronizer(final ClusterService clusterService, FeatureService featureService, QueryableBuiltInRolesProviderFactory rolesProviderFactory, NativeRolesStore nativeRolesStore, ReservedRolesStore reservedRolesStore, FileRolesStore fileRolesStore, ThreadPool threadPool) {
        this.clusterService = clusterService;
        this.featureService = featureService;
        this.rolesProvider = rolesProviderFactory.createProvider(reservedRolesStore, fileRolesStore);
        this.nativeRolesStore = nativeRolesStore;
        this.executor = threadPool.generic();
        this.markRolesAsSyncedTaskQueue = clusterService.createTaskQueue("mark-built-in-roles-as-synced-task-queue", Priority.LOW, MARK_ROLES_AS_SYNCED_TASK_EXECUTOR);
        this.rolesProvider.addListener(this::builtInRolesChanged);
        this.clusterService.addLifecycleListener(new LifecycleListener(){

            public void beforeStop() {
                clusterService.removeListener((ClusterStateListener)QueryableBuiltInRolesSynchronizer.this);
            }

            public void beforeStart() {
                clusterService.addListener((ClusterStateListener)QueryableBuiltInRolesSynchronizer.this);
            }
        });
    }

    private void builtInRolesChanged(QueryableBuiltInRoles roles) {
        logger.debug("Built-in roles changed, attempting to sync to .security index");
        ClusterState state = this.clusterService.state();
        if (this.shouldSyncBuiltInRoles(state)) {
            this.syncBuiltInRoles(roles);
        }
    }

    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState state = event.state();
        if (this.isSecurityIndexDeleted(event)) {
            this.securityIndexDeleted = true;
            logger.trace("Received security index deletion event, skipping built-in roles synchronization");
            return;
        }
        if (this.isSecurityIndexCreatedOrRecovered(event)) {
            this.securityIndexDeleted = false;
            logger.trace("Security index has been created/recovered, attempting to sync built-in roles");
        }
        if (this.shouldSyncBuiltInRoles(state)) {
            QueryableBuiltInRoles roles = this.rolesProvider.getRoles();
            this.syncBuiltInRoles(roles);
        }
    }

    public boolean isSynchronizationInProgress() {
        return this.synchronizationInProgress.get();
    }

    private void syncBuiltInRoles(QueryableBuiltInRoles roles) {
        if (this.synchronizationInProgress.compareAndSet(false, true)) {
            try {
                Map<String, String> indexedRolesDigests = this.readIndexedBuiltInRolesDigests(this.clusterService.state());
                if (roles.rolesDigest().equals(indexedRolesDigests)) {
                    logger.debug("Security index already contains the latest built-in roles indexed, skipping roles synchronization");
                    this.resetFailedSyncAttempts();
                    this.synchronizationInProgress.set(false);
                } else {
                    this.executor.execute(() -> this.doSyncBuiltinRoles(indexedRolesDigests, roles, (ActionListener<Void>)ActionListener.wrap(v -> {
                        logger.info("Successfully synced [{}] built-in roles to .security index", (Object)roles.roleDescriptors().size());
                        this.resetFailedSyncAttempts();
                        this.synchronizationInProgress.set(false);
                    }, e -> {
                        this.handleException((Exception)e);
                        this.synchronizationInProgress.set(false);
                    })));
                }
            }
            catch (Exception e) {
                logger.error("Failed to sync built-in roles", (Throwable)e);
                this.failedSyncAttempts.incrementAndGet();
                this.synchronizationInProgress.set(false);
            }
        }
    }

    private void handleException(Exception e) {
        boolean isUnexpectedFailure = false;
        if (e instanceof BulkRolesResponseException) {
            BulkRolesResponseException bulkException = (BulkRolesResponseException)e;
            boolean isBulkDeleteFailure = bulkException instanceof BulkDeleteRolesResponseException;
            for (Map.Entry<String, Exception> bulkFailure : bulkException.getFailures().entrySet()) {
                String logMessage = Strings.format((String)"Failed to [%s] built-in role [%s]", (Object[])new Object[]{isBulkDeleteFailure ? "delete" : "create/update", bulkFailure.getKey()});
                if (QueryableBuiltInRolesSynchronizer.isExpectedFailure(bulkFailure.getValue())) {
                    logger.info(logMessage, (Throwable)bulkFailure.getValue());
                    continue;
                }
                isUnexpectedFailure = true;
                logger.warn(logMessage, (Throwable)bulkFailure.getValue());
            }
        } else if (QueryableBuiltInRolesSynchronizer.isExpectedFailure(e)) {
            logger.info("Failed to sync built-in roles to .security index", (Throwable)e);
        } else {
            isUnexpectedFailure = true;
            logger.warn("Failed to sync built-in roles to .security index due to unexpected exception", (Throwable)e);
        }
        if (isUnexpectedFailure) {
            this.failedSyncAttempts.incrementAndGet();
        }
    }

    private void resetFailedSyncAttempts() {
        if (this.failedSyncAttempts.get() > 0) {
            logger.trace("resetting failed sync attempts to 0");
            this.failedSyncAttempts.set(0);
        }
    }

    int getFailedSyncAttempts() {
        return this.failedSyncAttempts.get();
    }

    private static boolean isExpectedFailure(Exception e) {
        Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
        return ExceptionsHelper.isNodeOrShardUnavailableTypeException((Throwable)cause) || TransportActions.isShardNotAvailableException((Throwable)cause) || cause instanceof IndexClosedException || cause instanceof IndexPrimaryShardNotAllocatedException || cause instanceof NotMasterException || cause instanceof ResourceAlreadyExistsException || cause instanceof VersionConflictEngineException || cause instanceof DocumentMissingException || cause instanceof FailedToMarkBuiltInRolesAsSyncedException || e instanceof FailedToCommitClusterStateException && "node closed".equals(cause.getMessage());
    }

    private static boolean isMixedVersionCluster(DiscoveryNodes nodes) {
        Version version = null;
        for (DiscoveryNode n : nodes) {
            if (version == null) {
                version = n.getVersion();
                continue;
            }
            if (version.equals((Object)n.getVersion())) continue;
            return true;
        }
        return false;
    }

    private boolean shouldSyncBuiltInRoles(ClusterState state) {
        if (!state.nodes().isLocalNodeElectedMaster()) {
            logger.trace("Local node is not the master, skipping built-in roles synchronization");
            return false;
        }
        if (this.failedSyncAttempts.get() >= 10) {
            logger.debug("Failed to sync built-in roles to .security index [{}] times. Skipping built-in roles synchronization.", (Object)this.failedSyncAttempts.get());
            return false;
        }
        if (!state.clusterRecovered()) {
            logger.trace("Cluster state has not recovered yet, skipping built-in roles synchronization");
            return false;
        }
        if (!this.nativeRolesStore.isEnabled()) {
            logger.trace("Native roles store is not enabled, skipping built-in roles synchronization");
            return false;
        }
        if (state.nodes().getDataNodes().isEmpty()) {
            logger.trace("No data nodes in the cluster, skipping built-in roles synchronization");
            return false;
        }
        if (QueryableBuiltInRolesSynchronizer.isMixedVersionCluster(state.nodes())) {
            logger.trace("Not all nodes are on the same version, skipping built-in roles synchronization");
            return false;
        }
        if (!this.featureService.clusterHasFeature(state, QUERYABLE_BUILT_IN_ROLES_FEATURE)) {
            logger.trace("Not all nodes support queryable built-in roles feature, skipping built-in roles synchronization");
            return false;
        }
        if (this.securityIndexDeleted) {
            logger.trace("Security index is deleted, skipping built-in roles synchronization");
            return false;
        }
        if (this.isSecurityIndexClosed(state)) {
            logger.trace("Security index is closed, skipping built-in roles synchronization");
            return false;
        }
        return true;
    }

    private void doSyncBuiltinRoles(Map<String, String> indexedRolesDigests, QueryableBuiltInRoles roles, ActionListener<Void> listener) {
        Set<RoleDescriptor> rolesToUpsert = QueryableBuiltInRolesUtils.determineRolesToUpsert(roles, indexedRolesDigests);
        Set<String> rolesToDelete = QueryableBuiltInRolesUtils.determineRolesToDelete(roles, indexedRolesDigests);
        assert (Sets.intersection(rolesToUpsert.stream().map(RoleDescriptor::getName).collect(Collectors.toSet()), rolesToDelete).isEmpty()) : "The roles to upsert and delete should not have any common roles";
        if (rolesToUpsert.isEmpty() && rolesToDelete.isEmpty()) {
            logger.debug("No changes to built-in roles to sync to .security index");
            listener.onResponse(null);
            return;
        }
        this.indexRoles(rolesToUpsert, (ActionListener<Void>)listener.delegateFailureAndWrap((l1, indexResponse) -> this.deleteRoles(rolesToDelete, (ActionListener<Void>)l1.delegateFailureAndWrap((l2, deleteResponse) -> this.markRolesAsSynced(indexedRolesDigests, roles.rolesDigest(), (ActionListener<Void>)l2)))));
    }

    private void deleteRoles(Set<String> rolesToDelete, ActionListener<Void> listener) {
        if (rolesToDelete.isEmpty()) {
            listener.onResponse(null);
            return;
        }
        this.nativeRolesStore.deleteRoles(rolesToDelete, WriteRequest.RefreshPolicy.IMMEDIATE, false, (ActionListener<BulkRolesResponse>)ActionListener.wrap(deleteResponse -> {
            Map<String, Exception> deleteFailure = deleteResponse.getItems().stream().filter(BulkRolesResponse.Item::isFailed).collect(Collectors.toMap(BulkRolesResponse.Item::getRoleName, BulkRolesResponse.Item::getCause));
            if (deleteFailure.isEmpty()) {
                listener.onResponse(null);
            } else {
                listener.onFailure((Exception)new BulkDeleteRolesResponseException(deleteFailure));
            }
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    private void indexRoles(Collection<RoleDescriptor> rolesToUpsert, ActionListener<Void> listener) {
        if (rolesToUpsert.isEmpty()) {
            listener.onResponse(null);
            return;
        }
        this.nativeRolesStore.putRoles(WriteRequest.RefreshPolicy.IMMEDIATE, rolesToUpsert, false, (ActionListener<BulkRolesResponse>)ActionListener.wrap(response -> {
            Map<String, Exception> indexFailures = response.getItems().stream().filter(BulkRolesResponse.Item::isFailed).collect(Collectors.toMap(BulkRolesResponse.Item::getRoleName, BulkRolesResponse.Item::getCause));
            if (indexFailures.isEmpty()) {
                listener.onResponse(null);
            } else {
                listener.onFailure((Exception)new BulkIndexRolesResponseException(indexFailures));
            }
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    private boolean isSecurityIndexDeleted(ClusterChangedEvent event) {
        IndexMetadata previousSecurityIndexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(event.previousState().metadata());
        IndexMetadata currentSecurityIndexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(event.state().metadata());
        return previousSecurityIndexMetadata != null && currentSecurityIndexMetadata == null;
    }

    private boolean isSecurityIndexCreatedOrRecovered(ClusterChangedEvent event) {
        IndexMetadata previousSecurityIndexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(event.previousState().metadata());
        IndexMetadata currentSecurityIndexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(event.state().metadata());
        return previousSecurityIndexMetadata == null && currentSecurityIndexMetadata != null;
    }

    private boolean isSecurityIndexClosed(ClusterState state) {
        IndexMetadata indexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(state.metadata());
        return indexMetadata != null && indexMetadata.getState() == IndexMetadata.State.CLOSE;
    }

    private void markRolesAsSynced(Map<String, String> expectedRolesDigests, Map<String, String> newRolesDigests, ActionListener<Void> listener) {
        IndexMetadata securityIndexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(this.clusterService.state().metadata());
        if (securityIndexMetadata == null) {
            listener.onFailure((Exception)new IndexNotFoundException(".security"));
            return;
        }
        Index concreteSecurityIndex = securityIndexMetadata.getIndex();
        this.markRolesAsSyncedTaskQueue.submitTask("mark built-in roles as synced task", (ClusterStateTaskListener)new MarkRolesAsSyncedTask((ActionListener<Map<String, String>>)listener.delegateFailureAndWrap((l, response) -> {
            if (!newRolesDigests.equals(response)) {
                logger.debug(() -> Strings.format((String)"Another master node most probably indexed a newer versions of built-in roles in the meantime. Expected: [%s], Actual: [%s]", (Object[])new Object[]{newRolesDigests, response}));
                l.onFailure((Exception)new FailedToMarkBuiltInRolesAsSyncedException("Failed to mark built-in roles as synced. The expected role digests have changed."));
            } else {
                l.onResponse(null);
            }
        }), concreteSecurityIndex.getName(), expectedRolesDigests, newRolesDigests), null);
    }

    private Map<String, String> readIndexedBuiltInRolesDigests(ClusterState state) {
        IndexMetadata indexMetadata = QueryableBuiltInRolesSynchronizer.resolveSecurityIndexMetadata(state.metadata());
        if (indexMetadata == null) {
            return null;
        }
        return indexMetadata.getCustomData(METADATA_QUERYABLE_BUILT_IN_ROLES_DIGEST_KEY);
    }

    private static IndexMetadata resolveSecurityIndexMetadata(Metadata metadata) {
        return SecurityIndexManager.resolveConcreteIndex(".security", metadata);
    }

    static {
        String propertyValue = System.getProperty("es.queryable_built_in_roles_enabled");
        if ("false".equals(propertyValue)) {
            QUERYABLE_BUILT_IN_ROLES_ENABLED = false;
        } else if (propertyValue == null || propertyValue.isEmpty() || "true".equals(propertyValue)) {
            QUERYABLE_BUILT_IN_ROLES_ENABLED = true;
        } else {
            throw new IllegalStateException("system property [es.queryable_built_in_roles_enabled] may only be set to [true] or [false], but was [" + propertyValue + "]");
        }
        QUERYABLE_BUILT_IN_ROLES_FEATURE = new NodeFeature("security.queryable_built_in_roles");
        MARK_ROLES_AS_SYNCED_TASK_EXECUTOR = new SimpleBatchedExecutor<MarkRolesAsSyncedTask, Map<String, String>>(){

            public Tuple<ClusterState, Map<String, String>> executeTask(MarkRolesAsSyncedTask task, ClusterState clusterState) {
                return task.execute(clusterState);
            }

            public void taskSucceeded(MarkRolesAsSyncedTask task, Map<String, String> value) {
                task.success(value);
            }
        };
    }

    private static abstract class BulkRolesResponseException
    extends RuntimeException {
        private final Map<String, Exception> failures;

        BulkRolesResponseException(String message, Map<String, Exception> failures) {
            super(message);
            assert (failures != null && !failures.isEmpty());
            this.failures = failures;
            failures.values().forEach(this::addSuppressed);
        }

        Map<String, Exception> getFailures() {
            return this.failures;
        }
    }

    private static class BulkDeleteRolesResponseException
    extends BulkRolesResponseException {
        BulkDeleteRolesResponseException(Map<String, Exception> failures) {
            super("Failed to bulk delete built-in roles", failures);
        }
    }

    private static class FailedToMarkBuiltInRolesAsSyncedException
    extends RuntimeException {
        FailedToMarkBuiltInRolesAsSyncedException(String message) {
            super(message);
        }
    }

    static class MarkRolesAsSyncedTask
    implements ClusterStateTaskListener {
        private final ActionListener<Map<String, String>> listener;
        private final String concreteSecurityIndexName;
        private final Map<String, String> expectedRoleDigests;
        private final Map<String, String> newRoleDigests;

        MarkRolesAsSyncedTask(ActionListener<Map<String, String>> listener, String concreteSecurityIndexName, @Nullable Map<String, String> expectedRoleDigests, @Nullable Map<String, String> newRoleDigests) {
            this.listener = listener;
            this.concreteSecurityIndexName = concreteSecurityIndexName;
            this.expectedRoleDigests = expectedRoleDigests;
            this.newRoleDigests = newRoleDigests;
        }

        public Map<String, String> getNewRoleDigests() {
            return this.newRoleDigests;
        }

        Tuple<ClusterState, Map<String, String>> execute(ClusterState state) {
            IndexMetadata indexMetadata = state.metadata().index(this.concreteSecurityIndexName);
            if (indexMetadata == null) {
                throw new IndexNotFoundException(this.concreteSecurityIndexName);
            }
            Map existingRoleDigests = indexMetadata.getCustomData(QueryableBuiltInRolesSynchronizer.METADATA_QUERYABLE_BUILT_IN_ROLES_DIGEST_KEY);
            if (Objects.equals(this.expectedRoleDigests, existingRoleDigests)) {
                IndexMetadata.Builder indexMetadataBuilder = IndexMetadata.builder((IndexMetadata)indexMetadata);
                if (this.newRoleDigests != null) {
                    indexMetadataBuilder.putCustom(QueryableBuiltInRolesSynchronizer.METADATA_QUERYABLE_BUILT_IN_ROLES_DIGEST_KEY, this.newRoleDigests);
                } else {
                    indexMetadataBuilder.removeCustom(QueryableBuiltInRolesSynchronizer.METADATA_QUERYABLE_BUILT_IN_ROLES_DIGEST_KEY);
                }
                indexMetadataBuilder.version(indexMetadataBuilder.version() + 1L);
                ImmutableOpenMap.Builder builder = ImmutableOpenMap.builder((Map)state.metadata().indices());
                builder.put((Object)this.concreteSecurityIndexName, (Object)indexMetadataBuilder.build());
                return new Tuple((Object)ClusterState.builder((ClusterState)state).metadata(Metadata.builder((Metadata)state.metadata()).indices((Map)builder.build()).build()).build(), this.newRoleDigests);
            }
            return new Tuple((Object)state, (Object)existingRoleDigests);
        }

        void success(Map<String, String> value) {
            this.listener.onResponse(value);
        }

        public void onFailure(Exception e) {
            this.listener.onFailure(e);
        }
    }

    private static class BulkIndexRolesResponseException
    extends BulkRolesResponseException {
        BulkIndexRolesResponseException(Map<String, Exception> failures) {
            super("Failed to bulk create/update built-in roles", failures);
        }
    }
}

