"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIntegrations = void 0;
var _reactQuery = require("@tanstack/react-query");
var _react = require("react");
var _filter_integrations = require("../utils/filter_integrations");
var _use_kibana = require("./use_kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INTEGRATIONS_URL = '/api/fleet/epm/packages';
const INTEGRATIONS_CALL_TIMEOUT = 2000;
const queryKey = ['integrations'];

/**
 * Retrieves integrations from the Fleet plugin endpoint /api/fleet/epm/packages.
 * The integrations are then filtered, and we only keep the installed ones,
 * with category threat_intel and excluding the ti_utils integration.
 * We cancel the query in case it's taking too long to not block the Indicators page for the user.
 */
const useIntegrations = ({
  enabled
}) => {
  const timeoutRef = (0, _react.useRef)();
  const {
    http
  } = (0, _use_kibana.useKibana)().services;

  // retrieving the list of integrations from the fleet plugin's endpoint
  const fetchIntegrations = (0, _react.useCallback)(context => http.get(INTEGRATIONS_URL, {
    version: '2023-10-31',
    signal: context.signal
  }), [http]);
  const query = (0, _reactQuery.useQuery)(queryKey, fetchIntegrations, {
    select: data => data ? (0, _filter_integrations.filterIntegrations)(data.items) : [],
    enabled
  });
  const queryClient = (0, _reactQuery.useQueryClient)();
  (0, _react.useEffect)(() => {
    // cancel slow integrations call to unblock the UI
    timeoutRef.current = setTimeout(() => {
      queryClient.cancelQueries(queryKey);
    }, INTEGRATIONS_CALL_TIMEOUT);
    return () => {
      if (!timeoutRef.current) {
        return;
      }
      clearTimeout(timeoutRef.current);
    };
  }, [queryClient]);
  return query;
};
exports.useIntegrations = useIntegrations;