"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SortByRunAtAndRetryAt = exports.RunningOrClaimingTaskWithExpiredRetryAt = exports.RecognizedTask = exports.OneOfTaskTypes = exports.InactiveTasks = exports.IdleTaskWithExpiredRunAt = exports.EnabledTask = void 0;
exports.claimSort = claimSort;
exports.getClaimSort = getClaimSort;
exports.tasksClaimedByOwner = tasksClaimedByOwner;
exports.tasksOfType = tasksOfType;
exports.tasksWithPartitions = tasksWithPartitions;
exports.updateFieldsAndMarkAsFailed = void 0;
var _task = require("../task");
var _query_clauses = require("./query_clauses");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function tasksOfType(taskTypes) {
  return {
    bool: {
      should: [...taskTypes].map(type => ({
        term: {
          'task.taskType': type
        }
      }))
    }
  };
}
function tasksClaimedByOwner(taskManagerId, ...taskFilters) {
  return (0, _query_clauses.mustBeAllOf)({
    term: {
      'task.ownerId': taskManagerId
    }
  }, {
    term: {
      'task.status': 'claiming'
    }
  }, ...taskFilters);
}
const IdleTaskWithExpiredRunAt = exports.IdleTaskWithExpiredRunAt = {
  bool: {
    must: [{
      term: {
        'task.status': 'idle'
      }
    }, {
      range: {
        'task.runAt': {
          lte: 'now'
        }
      }
    }]
  }
};
const InactiveTasks = exports.InactiveTasks = {
  bool: {
    must_not: [{
      bool: {
        should: [{
          term: {
            'task.status': 'running'
          }
        }, {
          term: {
            'task.status': 'claiming'
          }
        }],
        // needed since default value is 0 when there is a `must` in the `bool`
        minimum_should_match: 1,
        must: {
          range: {
            'task.retryAt': {
              gt: 'now'
            }
          }
        }
      }
    }]
  }
};
const EnabledTask = exports.EnabledTask = {
  bool: {
    must: [{
      term: {
        'task.enabled': true
      }
    }]
  }
};
const RecognizedTask = exports.RecognizedTask = {
  bool: {
    must_not: [{
      term: {
        'task.status': _task.TaskStatus.Unrecognized
      }
    }]
  }
};
const RunningOrClaimingTaskWithExpiredRetryAt = exports.RunningOrClaimingTaskWithExpiredRetryAt = {
  bool: {
    must: [{
      bool: {
        should: [{
          term: {
            'task.status': 'running'
          }
        }, {
          term: {
            'task.status': 'claiming'
          }
        }]
      }
    }, {
      range: {
        'task.retryAt': {
          lte: 'now'
        }
      }
    }]
  }
};
const SortByRunAtAndRetryAtScript = {
  _script: {
    type: 'number',
    order: 'asc',
    script: {
      lang: 'painless',
      source: `
if (doc['task.retryAt'].size()!=0) {
  return doc['task.retryAt'].value.toInstant().toEpochMilli();
}
if (doc['task.runAt'].size()!=0) {
  return doc['task.runAt'].value.toInstant().toEpochMilli();
}
    `
    }
  }
};
const SortByRunAtAndRetryAt = exports.SortByRunAtAndRetryAt = SortByRunAtAndRetryAtScript;
function getSortByPriority(definitions) {
  if (definitions.size() === 0) return;
  return {
    _script: {
      type: 'number',
      order: 'desc',
      script: {
        lang: 'painless',
        // Use priority if explicitly specified in task definition, otherwise default to 50 (Normal)
        // TODO: we could do this locally as well, but they may starve
        source: `
          String taskType = doc['task.taskType'].value;
          if (doc['task.priority'].size() != 0) {
            return doc['task.priority'].value;
          } else if (params.priority_map.containsKey(taskType)) {
            return params.priority_map[taskType];
          } else {
            return ${_task.TaskPriority.Normal};
          }
        `,
        params: {
          priority_map: definitions.getAllDefinitions().reduce((acc, taskDefinition) => {
            if (taskDefinition.priority) {
              acc[taskDefinition.type] = taskDefinition.priority;
            }
            return acc;
          }, {})
        }
      }
    }
  };
}

// getClaimSort() is used to generate sort bits for the ES query
// should align with claimSort() below
function getClaimSort(definitions) {
  const sortByPriority = getSortByPriority(definitions);
  if (!sortByPriority) return [SortByRunAtAndRetryAt];
  return [sortByPriority, SortByRunAtAndRetryAt];
}

// claimSort() is used to sort tasks returned from a claimer by priority and date.
// Kept here so it should align with getClaimSort() above.
// Returns a copy of the tasks passed in.
function claimSort(definitions, tasks) {
  const priorityMap = {};
  tasks.forEach(task => {
    const taskType = task.taskType;
    const priority = getPriority(definitions, taskType);
    priorityMap[taskType] = priority;
  });
  return tasks.slice().sort(compare);
  function compare(a, b) {
    var _priorityMap$a$taskTy, _priorityMap$b$taskTy, _ref, _a$retryAt$valueOf, _a$retryAt, _ref2, _b$retryAt$valueOf, _b$retryAt;
    // sort by priority, descending
    const priorityA = (_priorityMap$a$taskTy = priorityMap[a.taskType]) !== null && _priorityMap$a$taskTy !== void 0 ? _priorityMap$a$taskTy : _task.TaskPriority.Normal;
    const priorityB = (_priorityMap$b$taskTy = priorityMap[b.taskType]) !== null && _priorityMap$b$taskTy !== void 0 ? _priorityMap$b$taskTy : _task.TaskPriority.Normal;
    if (priorityA > priorityB) return -1;
    if (priorityA < priorityB) return 1;

    // then sort by retry/runAt, ascending
    const runA = (_ref = (_a$retryAt$valueOf = (_a$retryAt = a.retryAt) === null || _a$retryAt === void 0 ? void 0 : _a$retryAt.valueOf()) !== null && _a$retryAt$valueOf !== void 0 ? _a$retryAt$valueOf : a.runAt.valueOf()) !== null && _ref !== void 0 ? _ref : 0;
    const runB = (_ref2 = (_b$retryAt$valueOf = (_b$retryAt = b.retryAt) === null || _b$retryAt === void 0 ? void 0 : _b$retryAt.valueOf()) !== null && _b$retryAt$valueOf !== void 0 ? _b$retryAt$valueOf : b.runAt.valueOf()) !== null && _ref2 !== void 0 ? _ref2 : 0;
    if (runA < runB) return -1;
    if (runA > runB) return 1;
    return 0;
  }
}
function getPriority(definitions, taskType) {
  var _definitions$get$prio, _definitions$get;
  return (_definitions$get$prio = (_definitions$get = definitions.get(taskType)) === null || _definitions$get === void 0 ? void 0 : _definitions$get.priority) !== null && _definitions$get$prio !== void 0 ? _definitions$get$prio : _task.TaskPriority.Normal;
}
const updateFieldsAndMarkAsFailed = ({
  fieldUpdates,
  claimableTaskTypes,
  skippedTaskTypes,
  taskMaxAttempts
}) => {
  const setScheduledAtScript = `if(ctx._source.task.retryAt != null && ZonedDateTime.parse(ctx._source.task.retryAt).toInstant().toEpochMilli() < params.now) {
    ctx._source.task.scheduledAt=ctx._source.task.retryAt;
  } else {
    ctx._source.task.scheduledAt=ctx._source.task.runAt;
  }`;
  const markAsClaimingScript = `ctx._source.task.status = "claiming"; ${Object.keys(fieldUpdates).map(field => `ctx._source.task.${field}=params.fieldUpdates.${field};`).join(' ')}`;
  const setScheduledAtAndMarkAsClaimed = `${setScheduledAtScript}
    ${markAsClaimingScript}`;
  return {
    source: `
    if (params.claimableTaskTypes.contains(ctx._source.task.taskType)) {
      ${setScheduledAtAndMarkAsClaimed}
    } else {
      ctx.op = "noop";
    }`,
    lang: 'painless',
    params: {
      now: new Date().getTime(),
      fieldUpdates,
      claimableTaskTypes,
      skippedTaskTypes,
      taskMaxAttempts
    }
  };
};
exports.updateFieldsAndMarkAsFailed = updateFieldsAndMarkAsFailed;
const OneOfTaskTypes = (field, types) => {
  return {
    bool: {
      must: [{
        terms: {
          [field]: types
        }
      }]
    }
  };
};
exports.OneOfTaskTypes = OneOfTaskTypes;
function tasksWithPartitions(partitions) {
  return {
    bool: {
      filter: [{
        bool: {
          should: [{
            terms: {
              'task.partition': partitions
            }
          }, {
            bool: {
              must_not: [{
                exists: {
                  field: 'task.partition'
                }
              }]
            }
          }]
        }
      }]
    }
  };
}