"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isRequestApiKeyType = exports.getApiKeyFromRequest = exports.getApiKeyAndUserScope = exports.createApiKey = void 0;
var _lodash = require("lodash");
var _spacesUtils = require("@kbn/spaces-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getCredentialsFromRequest = request => {
  const authorizationHeaderValue = request.headers.authorization;
  if (!authorizationHeaderValue || typeof authorizationHeaderValue !== 'string') {
    return null;
  }
  const [scheme] = authorizationHeaderValue.split(/\s+/);
  return authorizationHeaderValue.substring(scheme.length + 1);
};
const isRequestApiKeyType = user => {
  return (user === null || user === void 0 ? void 0 : user.authentication_type) === 'api_key';
};
exports.isRequestApiKeyType = isRequestApiKeyType;
const getApiKeyFromRequest = request => {
  const credentials = getCredentialsFromRequest(request);
  if (credentials) {
    const apiKey = Buffer.from(credentials, 'base64').toString().split(':');
    return {
      id: apiKey[0],
      api_key: apiKey[1]
    };
  }
  return null;
};
exports.getApiKeyFromRequest = getApiKeyFromRequest;
const createApiKey = async (taskInstances, request, security) => {
  if (!(await security.authc.apiKeys.areAPIKeysEnabled())) {
    throw Error('API keys are not enabled, cannot create API key.');
  }
  const user = security.authc.getCurrentUser(request);
  if (!user) {
    throw Error('Cannot authenticate current user.');
  }
  const apiKeyByTaskIdMap = new Map();

  // If the user passed in their own API key, simply return it
  if (isRequestApiKeyType(user)) {
    const apiKeyCreateResult = getApiKeyFromRequest(request);
    if (!apiKeyCreateResult) {
      throw Error('Could not create API key.');
    }
    const {
      id,
      api_key: apiKey
    } = apiKeyCreateResult;
    taskInstances.forEach(task => {
      apiKeyByTaskIdMap.set(task.id, {
        apiKey: Buffer.from(`${id}:${apiKey}`).toString('base64'),
        apiKeyId: apiKeyCreateResult.id
      });
    });
    return apiKeyByTaskIdMap;
  }
  // If the user did not pass in their own API key, we need to create 1 key per task
  // type (due to naming requirements).
  const taskTypes = [...new Set(taskInstances.map(task => task.taskType))];
  const apiKeyByTaskTypeMap = new Map();
  for (const taskType of taskTypes) {
    const apiKeyCreateResult = await security.authc.apiKeys.grantAsInternalUser(request, {
      name: (0, _lodash.truncate)(`TaskManager: ${taskType} - ${user.username}`, {
        length: 256
      }),
      role_descriptors: {},
      metadata: {
        managed: true
      }
    });
    if (!apiKeyCreateResult) {
      throw Error('Could not create API key.');
    }
    const {
      id,
      api_key: apiKey
    } = apiKeyCreateResult;
    apiKeyByTaskTypeMap.set(taskType, {
      apiKey: Buffer.from(`${id}:${apiKey}`).toString('base64'),
      apiKeyId: apiKeyCreateResult.id
    });
  }

  // Assign each of the created API keys to the task ID
  taskInstances.forEach(task => {
    const encodedApiKeyResult = apiKeyByTaskTypeMap.get(task.taskType);
    if (encodedApiKeyResult) {
      apiKeyByTaskIdMap.set(task.id, encodedApiKeyResult);
    }
  });
  return apiKeyByTaskIdMap;
};
exports.createApiKey = createApiKey;
const getApiKeyAndUserScope = async (taskInstances, request, security) => {
  const apiKeyByTaskIdMap = await createApiKey(taskInstances, request, security);
  const space = (0, _spacesUtils.getSpaceIdFromPath)(request.url.pathname);
  const user = security.authc.getCurrentUser(request);
  const apiKeyAndUserScopeByTaskId = new Map();
  taskInstances.forEach(task => {
    const encodedApiKeyResult = apiKeyByTaskIdMap.get(task.id);
    if (encodedApiKeyResult) {
      apiKeyAndUserScopeByTaskId.set(task.id, {
        apiKey: encodedApiKeyResult.apiKey,
        userScope: {
          apiKeyId: encodedApiKeyResult.apiKeyId,
          spaceId: (space === null || space === void 0 ? void 0 : space.spaceId) || 'default',
          // Set apiKeyCreatedByUser to true if the user passed in their own API key, since we do
          // not want to invalidate a specific API key that was not created by the task manager
          apiKeyCreatedByUser: isRequestApiKeyType(user)
        }
      });
    }
  });
  return apiKeyAndUserScopeByTaskId;
};
exports.getApiKeyAndUserScope = getApiKeyAndUserScope;