"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.editPrivateLocationRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _get_private_locations = require("../../../synthetics_service/get_private_locations");
var _private_location_repository = require("../../../repositories/private_location_repository");
var _feature = require("../../../feature");
var _constants2 = require("../../../../common/constants");
var _helpers = require("./helpers");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EditPrivateLocationSchema = _configSchema.schema.object({
  label: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1
  })),
  tags: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
});
const EditPrivateLocationQuery = _configSchema.schema.object({
  locationId: _configSchema.schema.string()
});
const isPrivateLocationLabelChanged = (oldLabel, newLabel) => {
  return typeof newLabel === 'string' && oldLabel !== newLabel;
};
const isPrivateLocationChanged = ({
  privateLocation,
  newParams
}) => {
  const isLabelChanged = isPrivateLocationLabelChanged(privateLocation.attributes.label, newParams.label);
  const areTagsChanged = Array.isArray(newParams.tags) && (!privateLocation.attributes.tags || privateLocation.attributes.tags && !(0, _lodash.isEqual)(privateLocation.attributes.tags, newParams.tags));
  return isLabelChanged || areTagsChanged;
};
const checkPrivileges = async ({
  routeContext,
  monitorsSpaces
}) => {
  const {
    request,
    response,
    server
  } = routeContext;
  const checkSavedObjectsPrivileges = server.security.authz.checkSavedObjectsPrivilegesWithRequest(request);
  const {
    hasAllRequested
  } = await checkSavedObjectsPrivileges('saved_object:synthetics-monitor/bulk_update', monitorsSpaces);
  if (!hasAllRequested) {
    return response.forbidden({
      body: {
        message: _i18n.i18n.translate('xpack.synthetics.editPrivateLocation.forbidden', {
          defaultMessage: 'You do not have sufficient permissions to update monitors in all required spaces. This private location is used by monitors in spaces where you lack update privileges.'
        })
      }
    });
  }
};
const editPrivateLocationRoute = () => ({
  method: 'PUT',
  path: _constants2.SYNTHETICS_API_URLS.PRIVATE_LOCATIONS + '/{locationId}',
  validate: {},
  validation: {
    request: {
      body: EditPrivateLocationSchema,
      params: EditPrivateLocationQuery
    }
  },
  requiredPrivileges: [_feature.PRIVATE_LOCATION_WRITE_API],
  handler: async routeContext => {
    const {
      response,
      request,
      savedObjectsClient
    } = routeContext;
    const {
      locationId
    } = request.params;
    const {
      label: newLocationLabel,
      tags: newTags
    } = request.body;
    const repo = new _private_location_repository.PrivateLocationRepository(routeContext);
    try {
      const {
        filtersStr
      } = (0, _common.parseArrayFilters)({
        locations: [locationId]
      });
      const [existingLocation, monitorsInLocation] = await Promise.all([repo.getPrivateLocation(locationId), routeContext.monitorConfigRepository.findDecryptedMonitors({
        spaceId: _constants.ALL_SPACES_ID,
        filter: filtersStr
      })]);
      let newLocation;
      if (isPrivateLocationChanged({
        privateLocation: existingLocation,
        newParams: request.body
      })) {
        // This privileges check is done only when changing the label, because changing the label will update also the monitors in that location
        if (isPrivateLocationLabelChanged(existingLocation.attributes.label, newLocationLabel) && monitorsInLocation.length) {
          await checkPrivileges({
            routeContext,
            monitorsSpaces: monitorsInLocation.map(({
              namespaces
            }) => namespaces[0])
          });
        }
        newLocation = await repo.editPrivateLocation(locationId, {
          label: newLocationLabel || existingLocation.attributes.label,
          tags: newTags || existingLocation.attributes.tags
        });
        if (isPrivateLocationLabelChanged(existingLocation.attributes.label, newLocationLabel)) {
          await (0, _helpers.updatePrivateLocationMonitors)({
            locationId,
            newLocationLabel,
            allPrivateLocations: await (0, _get_private_locations.getPrivateLocations)(savedObjectsClient),
            routeContext,
            monitorsInLocation
          });
        }
      }
      return (0, _helpers.toClientContract)({
        ...existingLocation,
        attributes: {
          ...existingLocation.attributes,
          ...(newLocation ? newLocation.attributes : {})
        }
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return response.notFound({
          body: {
            message: `Private location with id ${locationId} does not exist.`
          }
        });
      }
      throw error;
    }
  }
});
exports.editPrivateLocationRoute = editPrivateLocationRoute;