"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeleteMonitorAPI = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pMap = _interopRequireDefault(require("p-map"));
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _saved_objects = require("../../../../common/types/saved_objects");
var _edit_monitor = require("../edit_monitor");
var _runtime_types = require("../../../../common/runtime_types");
var _monitor_upgrade_sender = require("../../telemetry/monitor_upgrade_sender");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DeleteMonitorAPI {
  constructor(routeContext) {
    (0, _defineProperty2.default)(this, "routeContext", void 0);
    (0, _defineProperty2.default)(this, "result", []);
    this.routeContext = routeContext;
  }
  async getMonitorsToDelete(monitorIds) {
    const result = [];
    await (0, _pMap.default)(monitorIds, async monitorId => {
      const monitor = await this.getMonitorToDelete(monitorId);
      if (monitor) {
        result.push(monitor);
      }
    }, {
      stopOnError: false
    });
    return result;
  }
  async getMonitorToDelete(monitorId) {
    const {
      spaceId,
      savedObjectsClient,
      server,
      monitorConfigRepository
    } = this.routeContext;
    try {
      const {
        normalizedMonitor
      } = await monitorConfigRepository.getDecrypted(monitorId, spaceId);
      return normalizedMonitor;
    } catch (e) {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        this.result.push({
          id: monitorId,
          deleted: false,
          error: `Monitor id ${monitorId} not found!`
        });
      } else {
        server.logger.error(`Failed to decrypt monitor to delete, monitor id: ${monitorId}`, {
          error: e
        });
        (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(server.logger, server.telemetry, {
          reason: `Failed to decrypt monitor to delete ${monitorId}`,
          message: e === null || e === void 0 ? void 0 : e.message,
          type: 'deletionError',
          code: e === null || e === void 0 ? void 0 : e.code,
          status: e.status,
          stackVersion: server.stackVersion
        });
        return await savedObjectsClient.get(_saved_objects.syntheticsMonitorSavedObjectType, monitorId);
      }
    }
  }
  async execute({
    monitorIds
  }) {
    const {
      response,
      server
    } = this.routeContext;
    const monitors = await this.getMonitorsToDelete(monitorIds);
    for (const monitor of monitors) {
      const err = await (0, _edit_monitor.validatePermissions)(this.routeContext, monitor.attributes.locations);
      if (err) {
        return {
          res: response.forbidden({
            body: {
              message: err
            }
          })
        };
      }
    }
    try {
      var _result$statuses;
      const {
        errors,
        result
      } = await this.deleteMonitorBulk({
        monitors
      });
      (_result$statuses = result.statuses) === null || _result$statuses === void 0 ? void 0 : _result$statuses.forEach(res => {
        this.result.push({
          id: res.id,
          deleted: res.success
        });
      });
      return {
        errors,
        result: this.result
      };
    } catch (error) {
      server.logger.error(`Unable to delete Synthetics monitor with error ${error.message}`, {
        error
      });
      throw error;
    }
  }
  async deleteMonitorBulk({
    monitors
  }) {
    const {
      server,
      spaceId,
      syntheticsMonitorClient
    } = this.routeContext;
    const {
      logger,
      telemetry,
      stackVersion
    } = server;
    try {
      const deleteSyncPromise = syntheticsMonitorClient.deleteMonitors(monitors.map(normalizedMonitor => ({
        ...normalizedMonitor.attributes,
        id: normalizedMonitor.attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID]
      })), spaceId);
      const deletePromise = this.routeContext.monitorConfigRepository.bulkDelete(monitors.map(monitor => ({
        id: monitor.id,
        type: monitor.type
      })));
      const [errors, result] = await Promise.all([deleteSyncPromise, deletePromise]);
      monitors.forEach(monitor => {
        (0, _monitor_upgrade_sender.sendTelemetryEvents)(logger, telemetry, (0, _monitor_upgrade_sender.formatTelemetryDeleteEvent)(monitor, stackVersion, new Date().toISOString(), Boolean(monitor.attributes[_runtime_types.ConfigKey.SOURCE_INLINE]), errors));
      });
      return {
        errors,
        result
      };
    } catch (e) {
      throw e;
    }
  }
}
exports.DeleteMonitorAPI = DeleteMonitorAPI;