"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaSavedObjectsSLORepository = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _Either = require("fp-ts/lib/Either");
var _lodash = require("lodash");
var _constants = require("../../common/constants");
var _errors = require("../errors");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class KibanaSavedObjectsSLORepository {
  constructor(soClient, logger) {
    this.soClient = soClient;
    this.logger = logger;
  }
  async create(slo) {
    await this.soClient.create(_saved_objects.SO_SLO_TYPE, toStoredSLO(slo));
    return slo;
  }
  async update(slo) {
    let existingSavedObjectId;
    const findResponse = await this.soClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      perPage: 1,
      filter: `slo.attributes.id:(${slo.id})`
    });
    if (findResponse.total === 1) {
      existingSavedObjectId = findResponse.saved_objects[0].id;
    }
    await this.soClient.create(_saved_objects.SO_SLO_TYPE, toStoredSLO(slo), {
      id: existingSavedObjectId,
      overwrite: true
    });
    return slo;
  }
  async findById(id) {
    const response = await this.soClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      page: 1,
      perPage: 1,
      filter: `slo.attributes.id:(${id})`
    });
    if (response.total === 0) {
      throw new _errors.SLONotFound(`SLO [${id}] not found`);
    }
    const slo = this.toSLO(response.saved_objects[0]);
    if (slo === undefined) {
      throw new Error('Invalid stored SLO');
    }
    return slo;
  }
  async deleteById(id, {
    ignoreNotFound = false
  }) {
    const response = await this.soClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      page: 1,
      perPage: 1,
      filter: `slo.attributes.id:(${id})`
    });
    if (response.total === 0) {
      if (ignoreNotFound) {
        return;
      }
      throw new _errors.SLONotFound(`SLO [${id}] not found`);
    }
    await this.soClient.delete(_saved_objects.SO_SLO_TYPE, response.saved_objects[0].id);
  }
  async findAllByIds(ids) {
    if (ids.length === 0) return [];
    const response = await this.soClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      page: 1,
      perPage: ids.length,
      filter: `slo.attributes.id:(${ids.join(' or ')})`
    });
    return response.saved_objects.map(slo => this.toSLO(slo)).filter(slo => slo !== undefined);
  }
  async search(search, pagination, options = {
    tags: []
  }) {
    const {
      includeOutdatedOnly,
      tags
    } = options;
    const filter = [];
    if (tags.length > 0) {
      filter.push(`slo.attributes.tags: (${tags.join(' OR ')})`);
    }
    if (!!includeOutdatedOnly) {
      filter.push(`slo.attributes.version < ${_constants.SLO_MODEL_VERSION}`);
    }
    const response = await this.soClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      page: pagination.page,
      perPage: pagination.perPage,
      search,
      searchFields: ['name'],
      ...(filter.length && {
        filter: filter.join(' AND ')
      }),
      sortField: 'id',
      sortOrder: 'asc'
    });
    return {
      total: response.total,
      perPage: response.per_page,
      page: response.page,
      results: response.saved_objects.map(savedObject => this.toSLO(savedObject)).filter(slo => slo !== undefined)
    };
  }
  toSLO(storedSLOObject) {
    var _storedSLO$groupBy, _storedSLO$version, _storedSLO$createdBy, _storedSLO$updatedBy;
    const storedSLO = storedSLOObject.attributes;
    const result = _sloSchema.sloDefinitionSchema.decode({
      ...storedSLO,
      // groupBy was added in 8.10.0
      groupBy: (_storedSLO$groupBy = storedSLO.groupBy) !== null && _storedSLO$groupBy !== void 0 ? _storedSLO$groupBy : _sloSchema.ALL_VALUE,
      // version was added in 8.12.0. This is a safeguard against SO migration issue.
      // if not present, we considered the version to be 1, e.g. not migrated.
      // We would need to call the _reset api on this SLO.
      version: (_storedSLO$version = storedSLO.version) !== null && _storedSLO$version !== void 0 ? _storedSLO$version : 1,
      // settings.preventInitialBackfill was added in 8.15.0
      settings: (0, _lodash.merge)({
        preventInitialBackfill: false,
        syncDelay: '1m',
        frequency: '1m'
      }, storedSLO.settings),
      createdBy: (_storedSLO$createdBy = storedSLO.createdBy) !== null && _storedSLO$createdBy !== void 0 ? _storedSLO$createdBy : storedSLOObject.created_by,
      updatedBy: (_storedSLO$updatedBy = storedSLO.updatedBy) !== null && _storedSLO$updatedBy !== void 0 ? _storedSLO$updatedBy : storedSLOObject.updated_by
    });
    if ((0, _Either.isLeft)(result)) {
      this.logger.debug(`Invalid stored SLO with id [${storedSLO.id}]`);
      return undefined;
    }
    return result.right;
  }
}
exports.KibanaSavedObjectsSLORepository = KibanaSavedObjectsSLORepository;
function toStoredSLO(slo) {
  return _sloSchema.sloDefinitionSchema.encode(slo);
}