"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.debounceAsync = debounceAsync;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Unlike lodash's debounce, which resolves intermediate calls with the most
 * recent value, this implementation waits to resolve intermediate calls until
 * the next invocation resolves.
 *
 * @param fn an async function
 *
 * @returns A debounced async function that resolves on the next invocation
 */
function debounceAsync(fn, intervalMs) {
  let timeoutId;
  let resolve;
  let promise = new Promise(_resolve => {
    resolve = _resolve;
  });
  return (...args) => {
    if (timeoutId) {
      clearTimeout(timeoutId);
    }
    timeoutId = setTimeout(async () => {
      resolve(await fn(...args));
      promise = new Promise(_resolve => {
        resolve = _resolve;
      });
    }, intervalMs);
    return promise;
  };
}