"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNotesRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _esQuery = require("@kbn/es-query");
var _constants = require("../../../../../common/notes/constants");
var _saved_object_mappings = require("../../saved_object_mappings");
var _constants2 = require("../../../../../common/constants");
var _utils = require("../../../detection_engine/routes/utils");
var _common = require("../../utils/common");
var _notes = require("../../saved_object/notes");
var _notes2 = require("../../saved_object_mappings/notes");
var _timeline = require("../../../../../common/api/timeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable complexity */
const getNotesRoute = (router, startServices) => {
  router.versioned.get({
    path: _constants2.NOTE_URL,
    security: {
      authz: {
        requiredPrivileges: ['notes_read']
      }
    },
    access: 'public'
  }).addVersion({
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_timeline.GetNotesRequestQuery)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    try {
      var _queryParams$document, _queryParams$savedObj, _queryParams$search, _queryParams$sortFiel, _ref;
      const queryParams = request.query;
      const frameworkRequest = await (0, _common.buildFrameworkRequest)(context, request);

      // if documentIds is provided, we will search for all the notes associated with the documentIds
      const documentIds = (_queryParams$document = queryParams.documentIds) !== null && _queryParams$document !== void 0 ? _queryParams$document : null;
      if (documentIds != null) {
        // search for multiple document ids (like retrieving all the notes for all the alerts within a table)
        if (Array.isArray(documentIds)) {
          const options = {
            type: _notes2.noteSavedObjectType,
            filter: _esQuery.nodeBuilder.or(documentIds.map(documentId => _esQuery.nodeBuilder.is(`${_notes2.noteSavedObjectType}.attributes.eventId`, documentId))),
            page: 1,
            perPage: _constants2.NOTES_PER_PAGE_HARD_LIMIT
          };
          const res = await (0, _notes.getAllSavedNote)(frameworkRequest, options);
          const body = res !== null && res !== void 0 ? res : {};
          return response.ok({
            body
          });
        }

        // searching for all the notes associated with a specific document id
        const options = {
          type: _notes2.noteSavedObjectType,
          filter: _esQuery.nodeBuilder.is(`${_notes2.noteSavedObjectType}.attributes.eventId`, documentIds),
          page: 1,
          perPage: _constants2.NOTES_PER_PAGE_HARD_LIMIT
        };
        const res = await (0, _notes.getAllSavedNote)(frameworkRequest, options);
        return response.ok({
          body: res !== null && res !== void 0 ? res : {}
        });
      }

      // if savedObjectIds is provided, we will search for all the notes associated with the savedObjectIds
      const savedObjectIds = (_queryParams$savedObj = queryParams.savedObjectIds) !== null && _queryParams$savedObj !== void 0 ? _queryParams$savedObj : null;
      if (savedObjectIds != null) {
        // search for multiple saved object ids
        if (Array.isArray(savedObjectIds)) {
          const options = {
            type: _notes2.noteSavedObjectType,
            hasReference: savedObjectIds.map(savedObjectId => ({
              type: _saved_object_mappings.timelineSavedObjectType,
              id: savedObjectId
            })),
            page: 1,
            perPage: _constants2.NOTES_PER_PAGE_HARD_LIMIT
          };
          const res = await (0, _notes.getAllSavedNote)(frameworkRequest, options);
          const body = res !== null && res !== void 0 ? res : {};
          return response.ok({
            body
          });
        }

        // searching for all the notes associated with a specific saved object id
        const options = {
          type: _notes2.noteSavedObjectType,
          hasReference: {
            type: _saved_object_mappings.timelineSavedObjectType,
            id: savedObjectIds
          },
          perPage: _constants2.NOTES_PER_PAGE_HARD_LIMIT
        };
        const res = await (0, _notes.getAllSavedNote)(frameworkRequest, options);
        const body = res !== null && res !== void 0 ? res : {};
        return response.ok({
          body
        });
      }

      // retrieving all the notes following the query parameters
      const perPage = queryParams !== null && queryParams !== void 0 && queryParams.perPage ? parseInt(queryParams.perPage, 10) : 10;
      const page = queryParams !== null && queryParams !== void 0 && queryParams.page ? parseInt(queryParams.page, 10) : 1;
      const search = (_queryParams$search = queryParams === null || queryParams === void 0 ? void 0 : queryParams.search) !== null && _queryParams$search !== void 0 ? _queryParams$search : undefined;
      const sortField = (_queryParams$sortFiel = queryParams === null || queryParams === void 0 ? void 0 : queryParams.sortField) !== null && _queryParams$sortFiel !== void 0 ? _queryParams$sortFiel : undefined;
      const sortOrder = (_ref = queryParams === null || queryParams === void 0 ? void 0 : queryParams.sortOrder) !== null && _ref !== void 0 ? _ref : undefined;
      const filter = queryParams === null || queryParams === void 0 ? void 0 : queryParams.filter;
      const options = {
        type: _notes2.noteSavedObjectType,
        perPage,
        page,
        search,
        sortField,
        sortOrder,
        filter
      };

      // we need to combine the associatedFilter with the filter query
      // we have to type case here because the filter is a string (from the schema) and that cannot be changed as it would be a breaking change
      const filterAsKueryNode = filter || '';
      const filterKueryNodeArray = [filterAsKueryNode];

      // retrieve all the notes created by a specific user
      // the createdByFilter value is the uuid of the user
      const createdByFilter = queryParams === null || queryParams === void 0 ? void 0 : queryParams.createdByFilter; // now uuid
      if (createdByFilter) {
        // because the notes createdBy property can be either full_name, email or username
        // see pickSaveNote (https://github.com/elastic/kibana/blob/main/x-pack/solutions/security/plugins/security_solution/server/lib/timeline/saved_object/notes/saved_object.ts#L302)
        // which uses the getUserDisplayName (https://github.com/elastic/kibana/blob/main/src/platform/packages/shared/kbn-user-profile-components/src/user_profile.ts#L138)
        const [_, {
          security
        }] = await startServices();
        const users = await security.userProfiles.bulkGet({
          uids: new Set([createdByFilter])
        });
        // once we retrieve the user by the uuid we can search all the notes that have the createdBy property with full_name, email or username values
        if (users && users.length > 0) {
          const {
            user: {
              email,
              full_name: fullName,
              username: userName
            }
          } = users[0];
          const createdByNodeArray = [];
          if (fullName) {
            createdByNodeArray.push(_esQuery.nodeBuilder.is(`${_notes2.noteSavedObjectType}.attributes.createdBy`, fullName));
          }
          if (userName) {
            createdByNodeArray.push(_esQuery.nodeBuilder.is(`${_notes2.noteSavedObjectType}.attributes.createdBy`, userName));
          }
          if (email) {
            createdByNodeArray.push(_esQuery.nodeBuilder.is(`${_notes2.noteSavedObjectType}.attributes.createdBy`, email));
          }
          filterKueryNodeArray.push(_esQuery.nodeBuilder.or(createdByNodeArray));
        } else {
          throw new Error(`User with uid ${createdByFilter} not found`);
        }
      }
      const associatedFilter = queryParams === null || queryParams === void 0 ? void 0 : queryParams.associatedFilter;
      if (associatedFilter) {
        // select documents that have or don't have a reference to an empty value
        // used in combination with hasReference (not associated with a timeline) or hasNoReference (associated with a timeline)
        const referenceToATimeline = {
          type: _saved_object_mappings.timelineSavedObjectType,
          id: ''
        };

        // select documents that don't have a value in the eventId field (not associated with a document)
        const emptyDocumentIdFilter = _esQuery.nodeBuilder.is(`${_notes2.noteSavedObjectType}.attributes.eventId`, '');
        switch (associatedFilter) {
          case _constants.AssociatedFilter.documentOnly:
            // select documents that have a reference to an empty saved object id (not associated with a timeline)
            // and have a value in the eventId field (associated with a document)
            options.hasReference = referenceToATimeline;
            filterKueryNodeArray.push(_esQuery.nodeTypes.function.buildNode('not', emptyDocumentIdFilter));
            break;
          case _constants.AssociatedFilter.savedObjectOnly:
            // select documents that don't have a reference to an empty saved object id (associated with a timeline)
            // and don't have a value in the eventId field (not associated with a document)
            options.hasNoReference = referenceToATimeline;
            filterKueryNodeArray.push(emptyDocumentIdFilter);
            break;
          case _constants.AssociatedFilter.documentAndSavedObject:
            // select documents that don't have a reference to an empty saved object id (associated with a timeline)
            // and have a value in the eventId field (associated with a document)
            options.hasNoReference = referenceToATimeline;
            filterKueryNodeArray.push(_esQuery.nodeTypes.function.buildNode('not', emptyDocumentIdFilter));
            break;
          case _constants.AssociatedFilter.orphan:
            // select documents that have a reference to an empty saved object id (not associated with a timeline)
            // and don't have a value in the eventId field (not associated with a document)
            options.hasReference = referenceToATimeline;
            // TODO we might want to also check for the existence of the eventId field, on top of getting eventId having empty values
            filterKueryNodeArray.push(emptyDocumentIdFilter);
            break;
        }
      }

      // combine all filters
      options.filter = _esQuery.nodeBuilder.and(filterKueryNodeArray);
      const res = await (0, _notes.getAllSavedNote)(frameworkRequest, options);
      const body = res !== null && res !== void 0 ? res : {};
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      const siemResponse = (0, _utils.buildSiemResponse)(response);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getNotesRoute = getNotesRoute;