"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRiskScoreMappings = exports.MAX_PER_PAGE = void 0;
var _std = require("@kbn/std");
var _fp = require("lodash/fp");
var _risk_engine_data_client = require("../risk_engine_data_client");
var _saved_object_configuration = require("../utils/saved_object_configuration");
var _risk_score_data_client = require("../../risk_score/risk_score_data_client");
var _saved_object = require("../saved_object");
var _helpers = require("../../risk_score/tasks/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_PER_PAGE = exports.MAX_PER_PAGE = 10_000;
const updateRiskScoreMappings = async ({
  auditLogger,
  logger,
  getStartServices,
  kibanaVersion
}) => {
  const [coreStart] = await getStartServices();
  const soClientKibanaUser = coreStart.savedObjects.createInternalRepository();

  // Get all installed Risk Engine Configurations
  const savedObjectsResponse = await soClientKibanaUser.find({
    type: _saved_object.riskEngineConfigurationTypeName,
    perPage: MAX_PER_PAGE,
    namespaces: ['*']
  });
  await (0, _std.asyncForEach)(savedObjectsResponse.saved_objects, async savedObject => {
    var _savedObject$attribut;
    const namespace = (0, _fp.first)(savedObject.namespaces); // We install one Risk Engine Configuration object per space

    if (!namespace) {
      logger.error('Unexpected saved object. Risk Score saved objects must have a namespace');
      return;
    }
    const newConfig = await (0, _saved_object_configuration.getDefaultRiskEngineConfiguration)({
      namespace
    });
    if (((_savedObject$attribut = savedObject.attributes._meta) === null || _savedObject$attribut === void 0 ? void 0 : _savedObject$attribut.mappingsVersion) !== newConfig._meta.mappingsVersion) {
      var _savedObject$attribut2;
      logger.info(`Starting Risk Score mappings update from version ${(_savedObject$attribut2 = savedObject.attributes._meta) === null || _savedObject$attribut2 === void 0 ? void 0 : _savedObject$attribut2.mappingsVersion} to version ${newConfig._meta.mappingsVersion} on namespace ${namespace}`);
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = (0, _helpers.buildScopedInternalSavedObjectsClientUnsafe)({
        coreStart,
        namespace
      });
      const riskEngineDataClient = new _risk_engine_data_client.RiskEngineDataClient({
        logger,
        kibanaVersion,
        esClient,
        namespace,
        soClient,
        auditLogger
      });
      const riskScoreDataClient = new _risk_score_data_client.RiskScoreDataClient({
        logger,
        kibanaVersion,
        esClient,
        namespace,
        soClient,
        auditLogger
      });
      await riskScoreDataClient.createOrUpdateRiskScoreIndexTemplate();
      await riskScoreDataClient.createOrUpdateRiskScoreLatestIndex();
      await riskScoreDataClient.createOrUpdateRiskScoreComponentTemplate();
      await riskScoreDataClient.rolloverRiskScoreTimeSeriesIndex();
      await riskEngineDataClient.updateConfiguration({
        _meta: {
          mappingsVersion: newConfig._meta.mappingsVersion
        }
      });
      logger.debug(`Risk score mappings updated to version ${newConfig._meta.mappingsVersion} on namespace ${namespace}`);
    }
  });
};
exports.updateRiskScoreMappings = updateRiskScoreMappings;