"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldRetentionEnrichPolicyStatus = exports.getFieldRetentionEnrichPolicyName = exports.executeFieldRetentionEnrichPolicy = exports.deleteFieldRetentionEnrichPolicy = exports.createFieldRetentionEnrichPolicy = void 0;
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFieldRetentionEnrichPolicyName = ({
  namespace,
  entityType,
  version
}) => {
  return `entity_store_field_retention_${entityType}_${namespace}_v${version}`;
};
exports.getFieldRetentionEnrichPolicyName = getFieldRetentionEnrichPolicyName;
const createFieldRetentionEnrichPolicy = async ({
  esClient,
  description,
  options
}) => {
  const enrichFields = description.dynamic ? ['*'] : description.fields.map(({
    destination
  }) => destination);
  return esClient.enrich.putPolicy({
    name: getFieldRetentionEnrichPolicyName({
      namespace: options.namespace,
      entityType: description.entityType,
      version: description.version
    }),
    match: {
      indices: (0, _utils.getEntitiesIndexName)(description.entityType, options.namespace),
      match_field: description.identityField,
      enrich_fields: enrichFields
    }
  });
};
exports.createFieldRetentionEnrichPolicy = createFieldRetentionEnrichPolicy;
const executeFieldRetentionEnrichPolicy = async ({
  esClient,
  entityType,
  version,
  logger,
  options
}) => {
  const name = getFieldRetentionEnrichPolicyName({
    namespace: options.namespace,
    entityType,
    version
  });
  try {
    await esClient.enrich.executePolicy({
      name
    });
    return {
      executed: true
    };
  } catch (e) {
    if (e.statusCode === 404) {
      return {
        executed: false
      };
    }
    logger.error(`Error executing field retention enrich policy for ${entityType}: ${e.message}`);
    throw e;
  }
};
exports.executeFieldRetentionEnrichPolicy = executeFieldRetentionEnrichPolicy;
const deleteFieldRetentionEnrichPolicy = async ({
  description,
  options,
  esClient,
  logger,
  attempts = 5,
  delayMs = 2000
}) => {
  const name = getFieldRetentionEnrichPolicyName({
    namespace: options.namespace,
    entityType: description.entityType,
    version: description.version
  });
  let currentAttempt = 1;
  while (currentAttempt <= attempts) {
    try {
      await esClient.enrich.deletePolicy({
        name
      }, {
        ignore: [404]
      });
      return;
    } catch (e) {
      // a 429 status code indicates that the enrich policy is being executed
      if (currentAttempt === attempts || e.statusCode !== 429) {
        logger.error(`Error deleting enrich policy ${name}: ${e.message} after ${currentAttempt} attempts`);
        throw e;
      }
      logger.info(`Enrich policy ${name} is being executed, waiting for it to finish before deleting`);
      await new Promise(resolve => setTimeout(resolve, delayMs));
      currentAttempt++;
    }
  }
};
exports.deleteFieldRetentionEnrichPolicy = deleteFieldRetentionEnrichPolicy;
const getFieldRetentionEnrichPolicyStatus = async ({
  definitionMetadata,
  esClient
}) => {
  const name = getFieldRetentionEnrichPolicyName(definitionMetadata);
  const policy = await esClient.enrich.getPolicy({
    name
  }, {
    ignore: [404]
  });
  const policies = policy.policies;
  return {
    installed: policies.length > 0,
    id: name,
    resource: _entity_analytics.EngineComponentResourceEnum.enrich_policy
  };
};
exports.getFieldRetentionEnrichPolicyStatus = getFieldRetentionEnrichPolicyStatus;