"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchActionRequests = void 0;
var _esQuery = require("@kbn/es-query");
var _stringify = require("../../../utils/stringify");
var _ = require("../..");
var _constants = require("../../../../../common/endpoint/constants");
var _utils = require("../../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches a list of Action Requests from the Endpoint action request index (not fleet)
 * @param logger
 * @param agentTypes
 * @param commands
 * @param elasticAgentIds
 * @param esClient
 * @param endDate
 * @param from
 * @param size
 * @param startDate
 * @param userIds
 * @param unExpiredOnly
 * @param types
 */
const fetchActionRequests = async ({
  logger,
  esClient,
  from = 0,
  size = 10,
  agentTypes,
  commands,
  elasticAgentIds,
  endDate,
  startDate,
  userIds,
  unExpiredOnly = false,
  types
}) => {
  var _actionRequests$hits, _actionRequests$hits$, _actionRequests$hits$2, _actionRequests$hits2;
  const additionalFilters = [];
  if (commands !== null && commands !== void 0 && commands.length) {
    additionalFilters.push({
      terms: {
        'data.command': commands
      }
    });
  }
  if (agentTypes !== null && agentTypes !== void 0 && agentTypes.length) {
    additionalFilters.push({
      terms: {
        input_type: agentTypes
      }
    });
  }
  if (elasticAgentIds !== null && elasticAgentIds !== void 0 && elasticAgentIds.length) {
    additionalFilters.push({
      terms: {
        agents: elasticAgentIds
      }
    });
  }
  if (unExpiredOnly) {
    additionalFilters.push({
      range: {
        expiration: {
          gte: 'now'
        }
      }
    });
  }
  const must = [{
    bool: {
      filter: [...(0, _.getDateFilters)({
        startDate,
        endDate
      }), ...additionalFilters]
    }
  }];
  if (userIds !== null && userIds !== void 0 && userIds.length) {
    const userIdsKql = userIds.map(userId => `user_id:${userId}`).join(' or ');
    const mustClause = (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(userIdsKql));
    must.push(mustClause);
  }
  const isNotASingleActionType = !types || types && types.length > 1;
  const actionsSearchQuery = {
    index: _constants.ENDPOINT_ACTIONS_INDEX,
    size,
    from,
    query: {
      bool: {
        must,
        ...(isNotASingleActionType ? {} : getActionTypeFilter(types[0]))
      }
    },
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }]
  };
  const actionRequests = await esClient.search(actionsSearchQuery, {
    ignore: [404]
  }).catch(_utils.catchAndWrapError);
  const total = (_actionRequests$hits = actionRequests.hits) === null || _actionRequests$hits === void 0 ? void 0 : (_actionRequests$hits$ = _actionRequests$hits.total) === null || _actionRequests$hits$ === void 0 ? void 0 : _actionRequests$hits$.value;
  logger.debug(`Searching for action requests found a total of [${total}] records using search query:\n${(0, _stringify.stringify)(actionsSearchQuery, 15)}`);
  return {
    data: ((_actionRequests$hits$2 = actionRequests === null || actionRequests === void 0 ? void 0 : (_actionRequests$hits2 = actionRequests.hits) === null || _actionRequests$hits2 === void 0 ? void 0 : _actionRequests$hits2.hits) !== null && _actionRequests$hits$2 !== void 0 ? _actionRequests$hits$2 : []).map(esHit => {
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      return esHit._source;
    }),
    size,
    from,
    total
  };
};

/** @internal */
exports.fetchActionRequests = fetchActionRequests;
const getActionTypeFilter = actionType => {
  return actionType === 'manual' ? {
    must_not: {
      exists: {
        field: 'data.alert_id'
      }
    }
  } : actionType === 'automated' ? {
    filter: {
      exists: {
        field: 'data.alert_id'
      }
    }
  } : {};
};