"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.applicationLinksUpdater = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _capabilities = require("../../common/lib/capabilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Dependencies to update the application links
 */

/**
 * The ApplicationLinksUpdater class stores the links recursive hierarchy and keeps
 * the value of the app links in sync with all the application components.
 * It can be updated using the `update` method.
 *
 * It's meant to be used as a singleton instance.
 */
class ApplicationLinksUpdater {
  constructor() {
    (0, _defineProperty2.default)(this, "linksSubject$", new _rxjs.BehaviorSubject([]));
    (0, _defineProperty2.default)(this, "normalizedLinksSubject$", new _rxjs.BehaviorSubject({}));
    /** Observable that stores the links recursive hierarchy */
    (0, _defineProperty2.default)(this, "links$", void 0);
    /** Observable that stores all the links indexed by `SecurityPageName` */
    (0, _defineProperty2.default)(this, "normalizedLinks$", void 0);
    this.links$ = this.linksSubject$.asObservable();
    this.normalizedLinks$ = this.normalizedLinksSubject$.asObservable();
  }

  /**
   * Updates the internal app links applying the filter by permissions
   */
  update(appLinksToUpdate, params) {
    const filteredAppLinks = this.filterAppLinks(appLinksToUpdate, params);
    this.linksSubject$.next(Object.freeze(filteredAppLinks));
    this.normalizedLinksSubject$.next(Object.freeze(this.getNormalizedLinks(filteredAppLinks)));
  }

  /**
   * Returns the current links value
   */
  getLinksValue() {
    return this.linksSubject$.getValue();
  }

  /**
   * Returns the current normalized links value
   */
  getNormalizedLinksValue() {
    return this.normalizedLinksSubject$.getValue();
  }

  /**
   * Creates the `NormalizedLinks` structure from a `LinkItem` array
   */
  getNormalizedLinks(currentLinks, parentId) {
    return currentLinks.reduce((normalized, {
      links,
      ...currentLink
    }) => {
      normalized[currentLink.id] = {
        ...currentLink,
        parentId
      };
      if (links && links.length > 0) {
        Object.assign(normalized, this.getNormalizedLinks(links, currentLink.id));
      }
      return normalized;
    }, {});
  }

  /**
   * Filters the app links based on the links configuration
   */
  filterAppLinks(appLinks, params) {
    const {
      experimentalFeatures,
      uiSettingsClient,
      capabilities,
      license,
      upselling
    } = params;
    return appLinks.reduce((acc, {
      links,
      ...appLinkInfo
    }) => {
      if (!this.isLinkExperimentalKeyAllowed(appLinkInfo, experimentalFeatures) || !this.isLinkUiSettingsAllowed(appLinkInfo, uiSettingsClient)) {
        return acc;
      }
      if (!(0, _capabilities.hasCapabilities)(capabilities, appLinkInfo.capabilities) || !this.isLinkLicenseAllowed(appLinkInfo, license)) {
        if (upselling.isPageUpsellable(appLinkInfo.id)) {
          acc.push({
            ...appLinkInfo,
            unauthorized: true
          });
        }
        return acc; // not adding sub-links for links that are not authorized
      }
      const resultAppLink = appLinkInfo;
      if (links) {
        const childrenLinks = this.filterAppLinks(links, params);
        if (childrenLinks.length > 0) {
          resultAppLink.links = childrenLinks;
        }
      }
      acc.push(resultAppLink);
      return acc;
    }, []);
  }

  /**
   * Check if the link is allowed based on the uiSettingsClient
   */
  isLinkUiSettingsAllowed(link, uiSettingsClient) {
    if (!link.uiSettingRequired) {
      return true;
    }
    if (typeof link.uiSettingRequired === 'string') {
      return uiSettingsClient.get(link.uiSettingRequired) === true;
    }
    if (typeof link.uiSettingRequired === 'object') {
      return uiSettingsClient.get(link.uiSettingRequired.key) === link.uiSettingRequired.value;
    }

    // unsupported uiSettingRequired type
    return false;
  }

  /**
   * Check if the link is allowed based on the experimental features
   */
  isLinkExperimentalKeyAllowed(link, experimentalFeatures) {
    if (link.hideWhenExperimentalKey && experimentalFeatures[link.hideWhenExperimentalKey]) {
      return false;
    }
    if (link.experimentalKey && !experimentalFeatures[link.experimentalKey]) {
      return false;
    }
    return true;
  }

  /**
   * Check if the link is allowed based on the license
   */
  isLinkLicenseAllowed(link, license) {
    var _link$licenseType;
    const linkLicenseType = (_link$licenseType = link.licenseType) !== null && _link$licenseType !== void 0 ? _link$licenseType : 'basic';
    if (license) {
      if (!license.hasAtLeast(linkLicenseType)) {
        return false;
      }
    } else if (linkLicenseType !== 'basic') {
      return false;
    }
    return true;
  }
}

// Create singleton instance of ApplicationLinksUpdater
const applicationLinksUpdater = exports.applicationLinksUpdater = new ApplicationLinksUpdater();