"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupFleetForEndpoint = exports.installOrUpgradeEndpointFleetPackage = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _usage_tracker = require("./usage_tracker");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calls the fleet setup APIs and then installs the latest Endpoint package
 * @param kbnClient
 * @param logger
 */
const setupFleetForEndpoint = exports.setupFleetForEndpoint = _usage_tracker.usageTracker.track('setupFleetForEndpoint', async (kbnClient, logger) => {
  const log = logger !== null && logger !== void 0 ? logger : (0, _utils.createToolingLogger)();
  log.debug(`setupFleetForEndpoint(): Setting up fleet for endpoint`);

  // Setup Fleet
  try {
    const setupResponse = await kbnClient.request({
      path: _common.SETUP_API_ROUTE,
      headers: {
        'Elastic-Api-Version': _common.API_VERSIONS.public.v1
      },
      method: 'POST'
    }).catch(_utils.wrapErrorAndRejectPromise);
    if (!setupResponse.data.isInitialized) {
      log.error(new Error(JSON.stringify(setupResponse.data, null, 2)));
      throw new Error('Initializing the ingest manager failed, existing');
    }
  } catch (error) {
    log.error(error);
    throw error;
  }

  // Setup Agents
  try {
    const setupResponse = await kbnClient.request({
      path: _common.AGENTS_SETUP_API_ROUTES.CREATE_PATTERN,
      method: 'POST',
      headers: {
        'elastic-api-version': _common.API_VERSIONS.public.v1
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
    if (!setupResponse.data.isInitialized) {
      log.error(new Error(JSON.stringify(setupResponse, null, 2)));
      throw new Error('Initializing Fleet failed');
    }
  } catch (error) {
    log.error(error);
    throw error;
  }

  // Install/upgrade the endpoint package
  try {
    await installOrUpgradeEndpointFleetPackage(kbnClient, log);
  } catch (error) {
    log.error(error);
    throw error;
  }
});

/**
 * Installs the Endpoint package (or upgrades it) in Fleet to the latest available in the registry
 *
 * @param kbnClient
 * @param logger
 */
const installOrUpgradeEndpointFleetPackage = exports.installOrUpgradeEndpointFleetPackage = _usage_tracker.usageTracker.track('installOrUpgradeEndpointFleetPackage', async (kbnClient, logger) => {
  logger.debug(`installOrUpgradeEndpointFleetPackage(): starting`);
  const updatePackages = async () => {
    const installEndpointPackageResp = await kbnClient.request({
      path: _common.EPM_API_ROUTES.BULK_INSTALL_PATTERN,
      method: 'POST',
      body: {
        packages: ['endpoint']
      },
      query: {
        prerelease: true
      },
      headers: {
        'elastic-api-version': _common.API_VERSIONS.public.v1
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
    logger.debug(`Fleet bulk install response:`, installEndpointPackageResp.data);
    const bulkResp = installEndpointPackageResp.data.items;
    if (bulkResp.length <= 0) {
      throw new _utils.EndpointDataLoadingError('Installing the Endpoint package failed, response was empty, existing', bulkResp);
    }
    const installResponse = bulkResp[0];
    logger.debug('package install response:', installResponse);
    if (isFleetBulkInstallError(installResponse)) {
      if (installResponse.error instanceof Error) {
        throw new _utils.EndpointDataLoadingError(`Installing the Endpoint package failed: ${installResponse.error.message}`, bulkResp);
      }

      // Ignore `409` (conflicts due to Concurrent install or upgrades of package) errors
      if (installResponse.statusCode !== 409) {
        throw new _utils.EndpointDataLoadingError(installResponse.error, bulkResp);
      }
    }
    return bulkResp[0];
  };
  return (0, _utils.retryOnError)(updatePackages, _utils.RETRYABLE_TRANSIENT_ERRORS, logger, 5, 10000);
});
function isFleetBulkInstallError(installResponse) {
  return 'error' in installResponse && installResponse.error !== undefined;
}