"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerPrivilegesRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const requiredPrivilegesMap = {
  ingest_pipelines: _constants.APP_CLUSTER_REQUIRED_PRIVILEGES,
  manage_processors: ['manage']
};
const extractMissingPrivileges = (privilegesObject = {}) => Object.keys(privilegesObject).reduce((privileges, privilegeName) => {
  if (!privilegesObject[privilegeName]) {
    privileges.push(privilegeName);
  }
  return privileges;
}, []);
const registerPrivilegesRoute = ({
  router,
  config
}) => {
  router.get({
    path: `${_constants.API_BASE_PATH}/privileges/{permissions_type}`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        permissions_type: _configSchema.schema.oneOf([_configSchema.schema.literal('ingest_pipelines'), _configSchema.schema.literal('manage_processors')])
      })
    }
  }, async (ctx, req, res) => {
    const permissionsType = req.params.permissions_type;
    const privilegesResult = {
      hasAllPrivileges: true,
      missingPrivileges: {
        cluster: []
      }
    };

    // Skip the privileges check if security is not enabled
    if (!config.isSecurityEnabled()) {
      return res.ok({
        body: privilegesResult
      });
    }
    const {
      client: clusterClient
    } = (await ctx.core).elasticsearch;
    const requiredPrivileges = requiredPrivilegesMap[permissionsType];
    const {
      has_all_requested: hasAllPrivileges,
      cluster
    } = await clusterClient.asCurrentUser.security.hasPrivileges({
      body: {
        cluster: requiredPrivileges
      }
    });
    if (!hasAllPrivileges) {
      privilegesResult.missingPrivileges.cluster = extractMissingPrivileges(cluster);
    }
    privilegesResult.hasAllPrivileges = hasAllPrivileges;
    return res.ok({
      body: privilegesResult
    });
  });
};
exports.registerPrivilegesRoute = registerPrivilegesRoute;