"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importDataProvider = importDataProvider;
var _fileUploadCommon = require("@kbn/file-upload-common");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function importDataProvider({
  asCurrentUser
}) {
  async function initializeImport(index, settings, mappings, ingestPipelines, existingIndex = false) {
    let createdIndex;
    const createdPipelineIds = [];
    const id = generateId();
    try {
      if (existingIndex) {
        await updateMappings(index, mappings);
      } else {
        await createIndex(index, settings, mappings);
      }
      createdIndex = index;

      // create the pipeline if one has been supplied
      if (ingestPipelines !== undefined) {
        for (const p of ingestPipelines) {
          if (p.pipeline === undefined) {
            createdPipelineIds.push(undefined);
            continue;
          }
          const resp = await createPipeline(p.id, p.pipeline);
          createdPipelineIds.push(p.id);
          if (resp.acknowledged !== true) {
            throw resp;
          }
        }
      }
      return {
        success: true,
        id,
        index: createdIndex,
        pipelineIds: createdPipelineIds
      };
    } catch (error) {
      var _createdIndex;
      return {
        success: false,
        id: id,
        index: (_createdIndex = createdIndex) !== null && _createdIndex !== void 0 ? _createdIndex : '',
        pipelineIds: createdPipelineIds,
        error: error.body !== undefined ? error.body : error
      };
    }
  }
  async function importData(index, ingestPipelineId, data) {
    const docCount = data.length;
    const pipelineId = ingestPipelineId;
    try {
      let failures = [];
      if (data.length) {
        const resp = await indexData(index, pipelineId, data);
        if (resp.success === false) {
          if (resp.ingestError) {
            // all docs failed, abort
            throw resp;
          } else {
            // some docs failed.
            // still report success but with a list of failures
            failures = resp.failures || [];
          }
        }
      }
      return {
        success: true,
        index,
        pipelineId,
        docCount,
        failures
      };
    } catch (error) {
      return {
        success: false,
        index,
        pipelineId,
        error: error.body !== undefined ? error.body : error,
        docCount,
        ingestError: error.ingestError,
        failures: error.failures || []
      };
    }
  }
  async function createIndex(index, settings, mappings) {
    const body = {
      mappings: {
        _meta: {
          created_by: _fileUploadCommon.INDEX_META_DATA_CREATED_BY
        },
        properties: mappings.properties
      }
    };
    if (settings && Object.keys(settings).length) {
      body.settings = settings;
    }
    await asCurrentUser.indices.create({
      index,
      body
    }, {
      maxRetries: 0
    });
  }
  async function updateMappings(index, mappings) {
    var _resp$index;
    const resp = await asCurrentUser.indices.getMapping({
      index
    });
    const existingMappings = (_resp$index = resp[index]) === null || _resp$index === void 0 ? void 0 : _resp$index.mappings;
    if (!(0, _lodash.isEqual)(existingMappings.properties, mappings.properties)) {
      await asCurrentUser.indices.putMapping({
        index,
        ...mappings
      });
    }
  }
  async function indexData(index, pipelineId, data) {
    try {
      const body = [];
      for (let i = 0; i < data.length; i++) {
        body.push({
          index: {}
        });
        body.push(data[i]);
      }
      const bulkRequest = {
        index,
        body
      };
      if (pipelineId !== undefined) {
        bulkRequest.pipeline = pipelineId;
      }
      const resp = await asCurrentUser.bulk(bulkRequest, {
        maxRetries: 0,
        requestTimeout: 3600000
      });
      if (resp.errors) {
        throw resp;
      } else {
        return {
          success: true,
          docs: data.length,
          failures: []
        };
      }
    } catch (error) {
      let failures = [];
      let ingestError = false;
      if (error.errors !== undefined && Array.isArray(error.items)) {
        // an expected error where some or all of the bulk request
        // docs have failed to be ingested.
        failures = getFailures(error.items, data);
      } else {
        // some other error has happened.
        ingestError = true;
      }
      return {
        success: false,
        error,
        docCount: data.length,
        failures,
        ingestError
      };
    }
  }
  async function createPipeline(id, pipeline) {
    return await asCurrentUser.ingest.putPipeline({
      id,
      body: pipeline
    });
  }
  function getFailures(items, data) {
    const failures = [];
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      if (item.index && item.index.error) {
        failures.push({
          item: i,
          reason: item.index.error.reason,
          caused_by: item.index.error.caused_by,
          doc: data[i]
        });
      }
    }
    return failures;
  }
  return {
    initializeImport,
    importData
  };
}
function generateId() {
  return Math.random().toString(36).substr(2, 9);
}