"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const server_sdk_1 = require("@openfeature/server-sdk");
const node_server_sdk_1 = require("@launchdarkly/node-server-sdk");
const translateContext_1 = __importDefault(require("./translateContext"));
const translateResult_1 = __importDefault(require("./translateResult"));
const translateTrackingEventDetails_1 = __importDefault(require("./translateTrackingEventDetails"));
const SafeLogger_1 = __importDefault(require("./SafeLogger"));
/**
 * Create a ResolutionDetails for an evaluation that produced a type different
 * than the expected type.
 * @param value The default value to populate the ResolutionDetails with.
 * @returns A ResolutionDetails with the default value.
 */
function wrongTypeResult(value) {
    return {
        value,
        reason: server_sdk_1.StandardResolutionReasons.ERROR,
        errorCode: server_sdk_1.ErrorCode.TYPE_MISMATCH,
    };
}
/**
 * An OpenFeature provider for the LaunchDarkly SDK for node.
 */
class LaunchDarklyProvider {
    initTimeoutSeconds;
    logger;
    client;
    clientConstructionError;
    metadata = {
        name: 'launchdarkly-node-provider',
    };
    runsOn = 'server';
    events = new server_sdk_1.OpenFeatureEventEmitter();
    /**
     * Construct a {@link LaunchDarklyProvider}.
     * @param sdkKey The SDK key.
     * @param options Any options for the SDK.
     * @param initTimeoutSeconds The default amount of time to wait for initialization in seconds.
     * Defaults to 10 seconds.
     */
    constructor(sdkKey, options = {}, initTimeoutSeconds = 10) {
        this.initTimeoutSeconds = initTimeoutSeconds;
        if (options.logger) {
            this.logger = new SafeLogger_1.default(options.logger, (0, node_server_sdk_1.basicLogger)({ level: 'info' }));
        }
        else {
            this.logger = (0, node_server_sdk_1.basicLogger)({ level: 'info' });
        }
        try {
            this.client = (0, node_server_sdk_1.init)(sdkKey, {
                ...options,
                wrapperName: 'open-feature-node-server',
                // The wrapper version should be kept on its own line to allow easy updates using
                // release-please.
                wrapperVersion: '0.6.0', // x-release-please-version
            });
            this.client.on('update', ({ key }) => this.events.emit(server_sdk_1.ProviderEvents.ConfigurationChanged, { flagsChanged: [key] }));
        }
        catch (e) {
            this.clientConstructionError = e;
            this.logger.error(`Encountered unrecoverable initialization error, ${e}`);
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async initialize(context) {
        if (!this.client) {
            // The client could not be constructed.
            if (this.clientConstructionError) {
                throw this.clientConstructionError;
            }
            throw new Error('Unknown problem encountered during initialization');
        }
        await this.client.waitForInitialization({ timeout: this.initTimeoutSeconds });
    }
    /**
     * Determines the boolean variation of a feature flag for a context, along with information about
     * how it was calculated.
     *
     * If the flag does not evaluate to a boolean value, then the defaultValue will be returned.
     *
     * @param flagKey The unique key of the feature flag.
     * @param defaultValue The default value of the flag, to be used if the value is not available
     *   from LaunchDarkly.
     * @param context The context requesting the flag. The client will generate an analytics event to
     *   register this context with LaunchDarkly if the context does not already exist.
     * @returns A promise which will resolve to a ResolutionDetails.
     */
    async resolveBooleanEvaluation(flagKey, defaultValue, context) {
        const res = await this.client.variationDetail(flagKey, this.translateContext(context), defaultValue);
        if (typeof res.value === 'boolean') {
            return (0, translateResult_1.default)(res);
        }
        return wrongTypeResult(defaultValue);
    }
    /**
     * Determines the string variation of a feature flag for a context, along with information about
     * how it was calculated.
     *
     * If the flag does not evaluate to a string value, then the defaultValue will be returned.
     *
     * @param flagKey The unique key of the feature flag.
     * @param defaultValue The default value of the flag, to be used if the value is not available
     *   from LaunchDarkly.
     * @param context The context requesting the flag. The client will generate an analytics event to
     *   register this context with LaunchDarkly if the context does not already exist.
     * @returns A promise which will resolve to a ResolutionDetails.
     */
    async resolveStringEvaluation(flagKey, defaultValue, context) {
        const res = await this.client.variationDetail(flagKey, this.translateContext(context), defaultValue);
        if (typeof res.value === 'string') {
            return (0, translateResult_1.default)(res);
        }
        return wrongTypeResult(defaultValue);
    }
    /**
     * Determines the numeric variation of a feature flag for a context, along with information about
     * how it was calculated.
     *
     * If the flag does not evaluate to a numeric value, then the defaultValue will be returned.
     *
     * @param flagKey The unique key of the feature flag.
     * @param defaultValue The default value of the flag, to be used if the value is not available
     *   from LaunchDarkly.
     * @param context The context requesting the flag. The client will generate an analytics event to
     *   register this context with LaunchDarkly if the context does not already exist.
     * @returns A promise which will resolve to a ResolutionDetails.
     */
    async resolveNumberEvaluation(flagKey, defaultValue, context) {
        const res = await this.client.variationDetail(flagKey, this.translateContext(context), defaultValue);
        if (typeof res.value === 'number') {
            return (0, translateResult_1.default)(res);
        }
        return wrongTypeResult(defaultValue);
    }
    /**
     * Determines the object variation of a feature flag for a context, along with information about
     * how it was calculated.
     *
     * @param flagKey The unique key of the feature flag.
     * @param defaultValue The default value of the flag, to be used if the value is not available
     *   from LaunchDarkly.
     * @param context The context requesting the flag. The client will generate an analytics event to
     *   register this context with LaunchDarkly if the context does not already exist.
     * @returns A promise which will resolve to a ResolutionDetails.
     */
    async resolveObjectEvaluation(flagKey, defaultValue, context) {
        const res = await this.client.variationDetail(flagKey, this.translateContext(context), defaultValue);
        if (typeof res.value === 'object') {
            return (0, translateResult_1.default)(res);
        }
        return wrongTypeResult(defaultValue);
    }
    // eslint-disable-next-line class-methods-use-this
    get hooks() {
        return [];
    }
    translateContext(context) {
        return (0, translateContext_1.default)(this.logger, context);
    }
    /**
     * Get the LDClient instance used by this provider.
     *
     * @returns The client for this provider.
     */
    getClient() {
        return this.client;
    }
    /**
     * Called by OpenFeature when it needs to close the provider. This will flush
     * events from the LDClient and then close it.
     */
    async onClose() {
        await this.client.flush();
        this.client.close();
    }
    /**
     * Track a user action or application state, usually representing a business objective or outcome.
     * @param trackingEventName The name of the event, which may correspond to a metric
     *   in Experimentation.
     * @param context The context to track.
     * @param trackingEventDetails Optional additional information to associate with the event.
     */
    track(trackingEventName, context, trackingEventDetails) {
        this.client.track(trackingEventName, this.translateContext(context), (0, translateTrackingEventDetails_1.default)(trackingEventDetails), trackingEventDetails?.value);
    }
}
exports.default = LaunchDarklyProvider;
//# sourceMappingURL=LaunchDarklyProvider.js.map