"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const InMemoryFeatureStore_1 = require("./InMemoryFeatureStore");
/**
 * This decorator can take a non-transactional {@link LDFeatureStore} implementation
 * and adapt it to be transactional through the use of an in-memory store acting as
 * cache.
 */
class TransactionalFeatureStore {
    constructor(_nonTransPersistenceStore) {
        this._nonTransPersistenceStore = _nonTransPersistenceStore;
        // persistence store is inital active store
        this._activeStore = this._nonTransPersistenceStore;
        this._memoryStore = new InMemoryFeatureStore_1.default();
    }
    get(kind, key, callback) {
        this._activeStore.get(kind, key, callback);
    }
    all(kind, callback) {
        this._activeStore.all(kind, callback);
    }
    init(allData, callback) {
        // adapt to applyChanges for common handling
        this.applyChanges(true, allData, callback);
    }
    delete(kind, key, version, callback) {
        // adapt to applyChanges for common handling
        const item = { key, version, deleted: true };
        this.applyChanges(false, {
            [kind.namespace]: {
                [key]: item,
            },
        }, callback);
    }
    upsert(kind, data, callback) {
        // adapt to applyChanges for common handling
        this.applyChanges(false, {
            [kind.namespace]: {
                [data.key]: data,
            },
        }, callback);
    }
    applyChanges(basis, data, callback, initMetadata, selector) {
        this._memoryStore.applyChanges(basis, data, () => {
            // TODO: SDK-1047 conditional propgation to persistence based on parameter
            if (basis) {
                // basis causes memory store to become the active store
                this._activeStore = this._memoryStore;
                this._nonTransPersistenceStore.init(data, callback);
            }
            else {
                const params = [];
                Object.entries(data).forEach(([namespace, items]) => {
                    Object.keys(items || {}).forEach((key) => {
                        params.push({ dataKind: { namespace }, item: Object.assign({ key }, items[key]) });
                    });
                });
                params
                    .reduce((previousPromise, nextParams) => previousPromise.then(() => new Promise((resolve) => {
                    this._nonTransPersistenceStore.upsert(nextParams.dataKind, nextParams.item, resolve);
                })), Promise.resolve())
                    .then(callback);
            }
        }, initMetadata, selector);
    }
    initialized(callback) {
        // this is valid because the active store will only switch to the in memory store
        // after it has already been initialized itself
        this._activeStore.initialized(callback);
    }
    close() {
        this._nonTransPersistenceStore.close();
        this._memoryStore.close();
    }
    getDescription() {
        return 'transactional persistent store';
    }
}
exports.default = TransactionalFeatureStore;
//# sourceMappingURL=TransactionalFeatureStore.js.map