"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLoadWatches = exports.useLoadWatchHistoryDetail = exports.useLoadWatchHistory = exports.useLoadWatchDetail = exports.useLoadSettings = exports.useGetWatchVisualizationData = exports.setHttpClient = exports.loadWatch = exports.loadIndexPatterns = exports.getMatchingIndices = exports.getHttpClient = exports.fetchFields = exports.executeWatch = exports.deleteWatches = exports.deactivateWatch = exports.createWatch = exports.activateWatch = exports.ackWatchAction = void 0;
var _settings = require("../models/settings");
var _watch = require("../models/watch");
var _watch_history_item = require("../models/watch_history_item");
var _watch_status = require("../models/watch_status");
var _use_request = require("./use_request");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

let httpClient;
const setHttpClient = anHttpClient => {
  httpClient = anHttpClient;
};
exports.setHttpClient = setHttpClient;
const getHttpClient = () => {
  return httpClient;
};
exports.getHttpClient = getHttpClient;
const basePath = _constants.ROUTES.API_ROOT;
const loadWatchesDeserializer = ({
  watches = []
}) => {
  return watches.map(watch => _watch.Watch.fromUpstreamJson(watch));
};
const useLoadWatches = pollIntervalMs => {
  return (0, _use_request.useRequest)({
    path: `${basePath}/watches`,
    method: 'get',
    pollIntervalMs,
    deserializer: loadWatchesDeserializer
  });
};
exports.useLoadWatches = useLoadWatches;
const loadWatchDetailDeserializer = ({
  watch = {}
}) => _watch.Watch.fromUpstreamJson(watch);
const useLoadWatchDetail = id => {
  return (0, _use_request.useRequest)({
    path: `${basePath}/watch/${id}`,
    method: 'get',
    deserializer: loadWatchDetailDeserializer
  });
};
exports.useLoadWatchDetail = useLoadWatchDetail;
const loadWatchHistoryDeserializer = ({
  watchHistoryItems = []
}) => {
  return watchHistoryItems.map(historyItem => _watch_history_item.WatchHistoryItem.fromUpstreamJson(historyItem));
};
const useLoadWatchHistory = (id, startTime) => {
  return (0, _use_request.useRequest)({
    query: startTime ? {
      startTime
    } : undefined,
    path: `${basePath}/watch/${id}/history`,
    method: 'get',
    deserializer: loadWatchHistoryDeserializer
  });
};
exports.useLoadWatchHistory = useLoadWatchHistory;
const loadWatchHistoryDetailDeserializer = ({
  watchHistoryItem
}) => _watch_history_item.WatchHistoryItem.fromUpstreamJson(watchHistoryItem);
const useLoadWatchHistoryDetail = id => {
  return (0, _use_request.useRequest)({
    path: !id ? '' : `${basePath}/history/${id}`,
    method: 'get',
    deserializer: loadWatchHistoryDetailDeserializer
  });
};
exports.useLoadWatchHistoryDetail = useLoadWatchHistoryDetail;
const deleteWatches = async watchIds => {
  const body = JSON.stringify({
    watchIds
  });
  const {
    results
  } = await getHttpClient().post(`${basePath}/watches/delete`, {
    body
  });
  return results;
};
exports.deleteWatches = deleteWatches;
const deactivateWatch = async id => {
  return (0, _use_request.sendRequest)({
    path: `${basePath}/watch/${id}/deactivate`,
    method: 'put'
  });
};
exports.deactivateWatch = deactivateWatch;
const activateWatch = async id => {
  return (0, _use_request.sendRequest)({
    path: `${basePath}/watch/${id}/activate`,
    method: 'put'
  });
};
exports.activateWatch = activateWatch;
const loadWatch = async id => {
  const {
    watch
  } = await getHttpClient().get(`${basePath}/watch/${id}`);
  return _watch.Watch.fromUpstreamJson(watch);
};
exports.loadWatch = loadWatch;
const getMatchingIndices = async pattern => {
  if (!pattern.startsWith('*')) {
    pattern = `*${pattern}`;
  }
  if (!pattern.endsWith('*')) {
    pattern = `${pattern}*`;
  }
  const body = JSON.stringify({
    pattern
  });
  const {
    indices
  } = await getHttpClient().post(`${basePath}/indices`, {
    body
  });
  return indices;
};
exports.getMatchingIndices = getMatchingIndices;
const fetchFields = async indexes => {
  const {
    fields
  } = await getHttpClient().post(`${basePath}/fields`, {
    body: JSON.stringify({
      indexes
    })
  });
  return fields;
};
exports.fetchFields = fetchFields;
const createWatch = async watch => {
  return await getHttpClient().put(`${basePath}/watch/${watch.id}`, {
    body: JSON.stringify(watch.upstreamJson)
  });
};
exports.createWatch = createWatch;
const executeWatch = async (executeWatchDetails, watch) => {
  return (0, _use_request.sendRequest)({
    path: `${basePath}/watch/execute`,
    method: 'put',
    body: JSON.stringify({
      executeDetails: executeWatchDetails.upstreamJson,
      watch: watch.upstreamJson
    })
  });
};
exports.executeWatch = executeWatch;
const loadIndexPatterns = async () => {
  return (0, _use_request.sendRequest)({
    path: `${basePath}/indices/index_patterns`,
    method: 'get'
  });
};
exports.loadIndexPatterns = loadIndexPatterns;
const getWatchVisualizationDataDeserializer = data => data === null || data === void 0 ? void 0 : data.visualizeData;
const useGetWatchVisualizationData = (watchModel, visualizeOptions) => {
  return (0, _use_request.useRequest)({
    path: `${basePath}/watch/visualize`,
    method: 'post',
    body: JSON.stringify({
      watch: watchModel.upstreamJson,
      options: visualizeOptions.upstreamJson
    }),
    deserializer: getWatchVisualizationDataDeserializer
  });
};
exports.useGetWatchVisualizationData = useGetWatchVisualizationData;
const loadSettingsDeserializer = data => _settings.Settings.fromUpstreamJson(data);
const useLoadSettings = () => {
  return (0, _use_request.useRequest)({
    path: `${basePath}/settings`,
    method: 'get',
    deserializer: loadSettingsDeserializer
  });
};
exports.useLoadSettings = useLoadSettings;
const ackWatchAction = async (watchId, actionId) => {
  const {
    watchStatus
  } = await getHttpClient().put(`${basePath}/watch/${watchId}/action/${actionId}/acknowledge`);
  return _watch_status.WatchStatus.fromUpstreamJson(watchStatus);
};
exports.ackWatchAction = ackWatchAction;