"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPalette = void 0;
var _color = _interopRequireDefault(require("color"));
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const Operators = {
  GTE: 'gte',
  GT: 'gt',
  LTE: 'lte',
  LT: 'lt'
};
const isValidColorRule = rule => {
  var _ref, _ref2;
  const {
    background_color: bColor,
    color: textColor
  } = rule;
  const {
    gauge
  } = rule;
  const {
    text
  } = rule;
  return Boolean(rule.operator && ((_ref = (_ref2 = bColor !== null && bColor !== void 0 ? bColor : textColor) !== null && _ref2 !== void 0 ? _ref2 : gauge) !== null && _ref !== void 0 ? _ref : text) && rule.value !== undefined);
};
const isMetricColorRule = rule => {
  var _metricRule$backgroun;
  const metricRule = rule;
  return ((_metricRule$backgroun = metricRule.background_color) !== null && _metricRule$backgroun !== void 0 ? _metricRule$backgroun : metricRule.color) ? true : false;
};
const isGaugeColorRule = rule => {
  const metricRule = rule;
  return Boolean(metricRule.gauge);
};
const getColor = rule => {
  if (isMetricColorRule(rule)) {
    var _rule$background_colo;
    return (_rule$background_colo = rule.background_color) !== null && _rule$background_colo !== void 0 ? _rule$background_colo : rule.color;
  } else if (isGaugeColorRule(rule)) {
    return rule.gauge;
  }
  return rule.text;
};
const getColorStopsWithMinMaxForAllGteOrWithLte = (rules, tailOperator, baseColor) => {
  const lastRule = rules[rules.length - 1];
  const lastRuleColor = getColor(lastRule);
  const initRules = baseColor ? [{
    stop: -Infinity,
    color: baseColor
  }] : [];
  const colorStops = rules.reduce((colors, rule, index, rulesArr) => {
    const rgbColor = getColor(rule);
    if (index === rulesArr.length - 1 && tailOperator === Operators.LTE) {
      return colors;
    }
    // if last operation is LTE, color of gte should be replaced by lte
    if (index === rulesArr.length - 2 && tailOperator === Operators.LTE) {
      return [...colors, {
        color: (0, _color.default)(lastRuleColor).hex(),
        stop: rule.value
      }];
    }
    return [...colors, {
      color: (0, _color.default)(rgbColor).hex(),
      stop: rule.value
    }];
  }, initRules);
  const stops = colorStops.reduce((prevStops, colorStop, index, colorStopsArr) => {
    if (index === colorStopsArr.length - 1) {
      return [...prevStops, {
        color: colorStop.color,
        stop: tailOperator === Operators.LTE ? lastRule.value : colorStop.stop + 1
      }];
    }
    return [...prevStops, {
      color: colorStop.color,
      stop: colorStopsArr[index + 1].stop
    }];
  }, []);
  const [rule] = rules;
  return {
    rangeMin: baseColor ? -Infinity : rule.value,
    rangeMax: tailOperator === Operators.LTE ? lastRule.value : Infinity,
    colorStops,
    stops,
    steps: colorStops.length,
    continuity: tailOperator === Operators.LTE ? baseColor ? 'below' : 'none' : baseColor ? 'all' : 'above'
  };
};
const getColorStopsWithMinMaxForLtWithLte = rules => {
  const lastRule = rules[rules.length - 1];
  const colorStops = rules.reduce((colors, rule, index, rulesArr) => {
    if (index === 0) {
      return [{
        color: (0, _color.default)(getColor(rule)).hex(),
        stop: -Infinity
      }];
    }
    const rgbColor = getColor(rule);
    return [...colors, {
      color: (0, _color.default)(rgbColor).hex(),
      stop: rulesArr[index - 1].value
    }];
  }, []);
  const stops = colorStops.reduce((prevStops, colorStop, index, colorStopsArr) => {
    if (index === colorStopsArr.length - 1) {
      return [...prevStops, {
        color: colorStop.color,
        stop: lastRule.value
      }];
    }
    return [...prevStops, {
      color: colorStop.color,
      stop: colorStopsArr[index + 1].stop
    }];
  }, []);
  return {
    rangeMin: -Infinity,
    rangeMax: lastRule.value,
    colorStops,
    stops,
    steps: colorStops.length + 1,
    continuity: 'below'
  };
};
const getColorStopWithMinMaxForLte = rule => {
  const colorStop = {
    color: (0, _color.default)(getColor(rule)).hex(),
    stop: rule.value
  };
  return {
    rangeMin: -Infinity,
    rangeMax: rule.value,
    colorStops: [colorStop],
    stops: [colorStop],
    steps: 1,
    continuity: 'below'
  };
};
const getColorStopWithMinMaxForGte = (rule, baseColor) => {
  const colorStop = {
    color: (0, _color.default)(getColor(rule)).hex(),
    stop: rule.value
  };
  return {
    colorStops: [...(baseColor ? [{
      color: baseColor,
      stop: -Infinity
    }] : []), colorStop],
    continuity: baseColor ? 'all' : 'above',
    rangeMax: Infinity,
    rangeMin: baseColor ? -Infinity : colorStop.stop,
    steps: 2,
    stops: [...(baseColor ? [{
      color: baseColor,
      stop: colorStop.stop
    }] : []), {
      color: colorStop.color,
      stop: Infinity
    }]
  };
};
const getCustomPalette = colorStopsWithMinMax => {
  return {
    name: 'custom',
    params: {
      continuity: 'all',
      maxSteps: 5,
      name: 'custom',
      progression: 'fixed',
      rangeMax: Infinity,
      rangeMin: -Infinity,
      rangeType: 'number',
      reverse: false,
      ...colorStopsWithMinMax
    },
    type: 'palette'
  };
};
const getPalette = (rules, baseColor) => {
  const validRules = rules.filter(rule => isValidColorRule(rule));
  validRules.sort((rule1, rule2) => {
    return rule1.value - rule2.value;
  });
  const kindOfRules = (0, _lodash.uniqBy)(validRules, 'operator');
  if (!kindOfRules.length) {
    return;
  }

  // lnsMetric is supporting lte only, if one rule is defined
  if (validRules.length === 1) {
    if (validRules[0].operator === Operators.LTE) {
      return getCustomPalette(getColorStopWithMinMaxForLte(validRules[0]));
    }
    if (validRules[0].operator === Operators.GTE) {
      return getCustomPalette(getColorStopWithMinMaxForGte(validRules[0], baseColor));
    }
    return;
  }
  const headRules = validRules.slice(0, -1);
  const tailRule = validRules[validRules.length - 1];
  const kindOfHeadRules = (0, _lodash.uniqBy)(headRules, 'operator');
  if (kindOfHeadRules.length > 1 || kindOfHeadRules[0].operator !== tailRule.operator && tailRule.operator !== Operators.LTE) {
    return;
  }
  const [rule] = kindOfHeadRules;
  if (rule.operator === Operators.LTE) {
    return;
  }
  if (rule.operator === Operators.LT) {
    if (tailRule.operator !== Operators.LTE) {
      return;
    }
    return getCustomPalette(getColorStopsWithMinMaxForLtWithLte(validRules));
  }
  if (rule.operator === Operators.GTE) {
    return getCustomPalette(getColorStopsWithMinMaxForAllGteOrWithLte(validRules, tailRule.operator, baseColor));
  }
};
exports.getPalette = getPalette;