"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTimelionRequestHandler = getTimelionRequestHandler;
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/data-plugin/public");
var _visualizationUtils = require("@kbn/visualization-utils");
var _plugin_services = require("./plugin_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getTimelionRequestHandler({
  uiSettings,
  http,
  timefilter,
  expressionAbortSignal
}) {
  const timezone = (0, _visualizationUtils.getTimeZone)(uiSettings);
  return async function ({
    timeRange,
    filters,
    query,
    visParams,
    searchSessionId,
    executionContext
  }) {
    var _filters$;
    const dataSearch = (0, _plugin_services.getDataSearch)();
    const expression = visParams.expression;
    const abortController = new AbortController();
    const expressionAbortHandler = function () {
      abortController.abort();
    };
    expressionAbortSignal.addEventListener('abort', expressionAbortHandler);
    if (!expression) {
      throw new Error(_i18n.i18n.translate('timelion.emptyExpressionErrorMessage', {
        defaultMessage: 'Timelion error: No expression provided'
      }));
    }
    let dataView;
    const firstFilterIndex = (_filters$ = filters[0]) === null || _filters$ === void 0 ? void 0 : _filters$.meta.index;
    if (firstFilterIndex) {
      dataView = await (0, _plugin_services.getIndexPatterns)().get(firstFilterIndex).catch(() => undefined);
    }
    const esQueryConfigs = (0, _public.getEsQueryConfig)(uiSettings);
    const doSearch = async searchOptions => {
      return await http.post('/internal/timelion/run', {
        body: JSON.stringify({
          sheet: [expression],
          extended: {
            es: {
              filter: (0, _esQuery.buildEsQuery)(dataView, query, filters, esQueryConfigs)
            }
          },
          time: {
            from: timeRangeBounds.min,
            to: timeRangeBounds.max,
            interval: visParams.interval,
            timezone
          },
          ...(searchOptions ? {
            searchSession: searchOptions
          } : {})
        }),
        context: executionContext,
        signal: abortController.signal
      });
    };

    // parse the time range client side to make sure it behaves like other charts
    const timeRangeBounds = timefilter.calculateBounds(timeRange);
    const searchTracker = dataSearch.session.isCurrentSession(searchSessionId) ? dataSearch.session.trackSearch({
      abort: () => abortController.abort(),
      poll: async () => {
        // don't use, keep this empty, onSavingSession is used instead
      },
      onSavingSession: async searchSessionOptions => {
        await doSearch(searchSessionOptions);
      }
    }) : undefined;
    try {
      const searchSessionOptions = dataSearch.session.getSearchOptions(searchSessionId);
      const visData = await doSearch(searchSessionOptions);
      searchTracker === null || searchTracker === void 0 ? void 0 : searchTracker.complete();
      return visData;
    } catch (e) {
      searchTracker === null || searchTracker === void 0 ? void 0 : searchTracker.error();
      if (e && e.body) {
        const err = new Error(`${_i18n.i18n.translate('timelion.requestHandlerErrorTitle', {
          defaultMessage: 'Timelion request error'
        })}:${e.body.title ? ' ' + e.body.title : ''} ${e.body.message}`);
        err.stack = e.stack;
        throw err;
      } else {
        throw e;
      }
    } finally {
      expressionAbortSignal.removeEventListener('abort', expressionAbortHandler);
    }
  };
}