"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createFormattedTable = void 0;
var _lodash = require("lodash");
var _services = require("../services");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const createFormattedTable = (table, visConfig) => {
  const {
    buckets,
    metrics
  } = visConfig;
  const formattedColumns = table.columns.reduce((acc, col, i) => {
    var _dimension$format$par;
    const isBucket = buckets === null || buckets === void 0 ? void 0 : buckets.find(({
      accessor
    }) => accessor === i);
    const dimension = isBucket || metrics.find(({
      accessor
    }) => accessor === i);
    if (!dimension) return acc;
    const formatter = (0, _services.getFormatService)().deserialize(dimension.format);
    const formattedColumn = {
      title: col.name,
      formatter,
      filterable: !!isBucket
    };
    const isDate = dimension.format.id === 'date' || ((_dimension$format$par = dimension.format.params) === null || _dimension$format$par === void 0 ? void 0 : _dimension$format$par.id) === 'date';
    const allowsNumericalAggregations = formatter.allowsNumericalAggregations;
    if (allowsNumericalAggregations || isDate || visConfig.totalFunc === _common.AggTypes.COUNT) {
      const sumOfColumnValues = table.rows.reduce((prev, curr) => {
        // some metrics return undefined for some of the values
        // derivative is an example of this as it returns undefined in the first row
        if (curr[col.id] === undefined) return prev;
        return prev + curr[col.id];
      }, 0);
      formattedColumn.sumTotal = sumOfColumnValues;
      switch (visConfig.totalFunc) {
        case _common.AggTypes.SUM:
          {
            if (!isDate) {
              formattedColumn.formattedTotal = formatter.convert(sumOfColumnValues);
              formattedColumn.total = sumOfColumnValues;
            }
            break;
          }
        case _common.AggTypes.AVG:
          {
            if (!isDate) {
              const total = sumOfColumnValues / table.rows.length;
              formattedColumn.formattedTotal = formatter.convert(total);
              formattedColumn.total = total;
            }
            break;
          }
        case _common.AggTypes.MIN:
          {
            const total = (0, _lodash.chain)(table.rows).map(col.id).min().value();
            formattedColumn.formattedTotal = formatter.convert(total);
            formattedColumn.total = total;
            break;
          }
        case _common.AggTypes.MAX:
          {
            const total = (0, _lodash.chain)(table.rows).map(col.id).max().value();
            formattedColumn.formattedTotal = formatter.convert(total);
            formattedColumn.total = total;
            break;
          }
        case _common.AggTypes.COUNT:
          {
            const total = table.rows.length;
            formattedColumn.formattedTotal = total;
            formattedColumn.total = total;
            break;
          }
        default:
          break;
      }
    }
    acc[col.id] = formattedColumn;
    return acc;
  }, {});
  return {
    // filter out columns which are not dimensions
    columns: table.columns.filter(col => formattedColumns[col.id]),
    rows: table.rows,
    formattedColumns
  };
};
exports.createFormattedTable = createFormattedTable;