"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEditorStateReducer = void 0;
exports.initEditorState = initEditorState;
var _lodash = require("lodash");
var _public = require("@kbn/data-plugin/public");
var _constants = require("./constants");
var _agg_group_helper = require("../../agg_group_helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initEditorState(vis) {
  return {
    ...vis.clone()
  };
}
const createEditorStateReducer = ({
  aggs: {
    createAggConfigs
  }
}) => (state, action) => {
  switch (action.type) {
    case _constants.EditorStateActionTypes.ADD_NEW_AGG:
      {
        const {
          schema
        } = action.payload;
        const defaultConfig = !state.data.aggs.aggs.find(agg => agg.schema === schema.name) && schema.defaults ? schema.defaults.slice(0, schema.max) : {
          schema: schema.name
        };
        const aggConfig = state.data.aggs.createAggConfig(defaultConfig, {
          addToAggConfigs: false
        });
        aggConfig.brandNew = true;
        const newAggs = [...state.data.aggs.aggs, aggConfig];
        return {
          ...state,
          data: {
            ...state.data,
            aggs: createAggConfigs(state.data.indexPattern, newAggs)
          }
        };
      }
    case _constants.EditorStateActionTypes.DISCARD_CHANGES:
      {
        return initEditorState(action.payload.vis);
      }
    case _constants.EditorStateActionTypes.CHANGE_AGG_TYPE:
      {
        const {
          aggId,
          value
        } = action.payload;
        const newAggs = state.data.aggs.aggs.map(agg => {
          if (agg.id === aggId) {
            agg.type = value;
            return agg.serialize();
          }
          return agg;
        });
        return {
          ...state,
          data: {
            ...state.data,
            aggs: createAggConfigs(state.data.indexPattern, newAggs)
          }
        };
      }
    case _constants.EditorStateActionTypes.SET_AGG_PARAM_VALUE:
      {
        const {
          aggId,
          paramName,
          value
        } = action.payload;
        const newAggs = state.data.aggs.aggs.map(agg => {
          if (agg.id === aggId) {
            const parsedAgg = agg.serialize();
            return {
              ...parsedAgg,
              params: {
                ...parsedAgg.params,
                [paramName]: value
              }
            };
          }
          return agg;
        });
        return {
          ...state,
          data: {
            ...state.data,
            aggs: createAggConfigs(state.data.indexPattern, newAggs)
          }
        };
      }
    case _constants.EditorStateActionTypes.SET_STATE_PARAM_VALUE:
      {
        const {
          paramName,
          value
        } = action.payload;
        return {
          ...state,
          params: {
            ...state.params,
            [paramName]: value
          }
        };
      }
    case _constants.EditorStateActionTypes.REMOVE_AGG:
      {
        let isMetric = false;
        const newAggs = state.data.aggs.aggs.filter(({
          id,
          schema
        }) => {
          if (id === action.payload.aggId) {
            const schemaDef = action.payload.schemas.find(s => s.name === schema);
            if (schemaDef && schemaDef.group === _public.AggGroupNames.Metrics) {
              isMetric = true;
            }
            return false;
          }
          return true;
        });
        if (isMetric && (0, _agg_group_helper.getEnabledMetricAggsCount)(newAggs) === 0) {
          const aggToEnable = newAggs.find(agg => agg.schema === 'metric');
          if (aggToEnable) {
            aggToEnable.enabled = true;
          }
        }
        return {
          ...state,
          data: {
            ...state.data,
            aggs: createAggConfigs(state.data.indexPattern, newAggs)
          }
        };
      }
    case _constants.EditorStateActionTypes.REORDER_AGGS:
      {
        const {
          sourceAgg,
          destinationAgg
        } = action.payload;
        const destinationIndex = state.data.aggs.aggs.indexOf(destinationAgg);
        const newAggs = [...state.data.aggs.aggs];
        newAggs.splice(destinationIndex, 0, newAggs.splice(state.data.aggs.aggs.indexOf(sourceAgg), 1)[0]);
        return {
          ...state,
          data: {
            ...state.data,
            aggs: createAggConfigs(state.data.indexPattern, newAggs)
          }
        };
      }
    case _constants.EditorStateActionTypes.TOGGLE_ENABLED_AGG:
      {
        const {
          aggId,
          enabled
        } = action.payload;
        const newAggs = state.data.aggs.aggs.map(agg => {
          if (agg.id === aggId) {
            const parsedAgg = agg.serialize();
            return {
              ...parsedAgg,
              enabled
            };
          }
          return agg;
        });
        return {
          ...state,
          data: {
            ...state.data,
            aggs: createAggConfigs(state.data.indexPattern, newAggs)
          }
        };
      }
    case _constants.EditorStateActionTypes.UPDATE_STATE_PARAMS:
      {
        const {
          params
        } = action.payload;
        return {
          ...state,
          params: (0, _lodash.cloneDeep)(params)
        };
      }
  }
};
exports.createEditorStateReducer = createEditorStateReducer;