"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EMPTY_STRING = void 0;
exports.getInitModelList = getInitModelList;
exports.getNextModel = getNextModel;
exports.getRange = getRange;
exports.getValidatedModels = getValidatedModels;
exports.hasInvalidValues = hasInvalidValues;
exports.parse = parse;
exports.validateValue = validateValue;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _eui = require("@elastic/eui");
var _range = require("./range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const EMPTY_STRING = exports.EMPTY_STRING = '';
const defaultRange = (0, _range.parseRange)('[0,Infinity)');
const generateId = (0, _eui.htmlIdGenerator)();
const defaultModel = {
  value: 0,
  id: generateId(),
  isInvalid: false
};
function parse(value) {
  const parsedValue = parseFloat(value);
  return isNaN(parsedValue) ? EMPTY_STRING : parsedValue;
}
function getRange(range) {
  try {
    return range ? (0, _range.parseRange)(range) : defaultRange;
  } catch (e) {
    throw new TypeError('Unable to parse range: ' + e.message);
  }
}
function validateValue(value, numberRange) {
  const result = {
    isInvalid: false
  };
  if (value === EMPTY_STRING) {
    result.isInvalid = true;
    result.error = EMPTY_STRING;
  } else if (!numberRange.within(value)) {
    result.isInvalid = true;
    result.error = _i18n.i18n.translate('visDefaultEditor.controls.numberList.invalidRangeErrorMessage', {
      defaultMessage: 'The value should be in the range of {min} to {max}.',
      values: {
        min: numberRange.min,
        max: numberRange.max
      }
    });
  }
  return result;
}
function validateValueAscending(inputValue, index, list) {
  const result = {
    isInvalidOrder: false
  };
  const previousModel = list[index - 1];
  if (previousModel !== undefined && inputValue !== undefined && inputValue !== '' && inputValue <= previousModel) {
    result.isInvalidOrder = true;
    result.error = _i18n.i18n.translate('visDefaultEditor.controls.numberList.invalidAscOrderErrorMessage', {
      defaultMessage: 'Value is not in ascending order.'
    });
  }
  return result;
}
function validateValueUnique(inputValue, index, list) {
  const result = {
    isDuplicate: false
  };
  if (inputValue !== EMPTY_STRING && list.indexOf(inputValue) !== index) {
    result.isDuplicate = true;
    result.error = _i18n.i18n.translate('visDefaultEditor.controls.numberList.duplicateValueErrorMessage', {
      defaultMessage: 'Duplicate value.'
    });
  }
  return result;
}
function getNextModel(list, range) {
  const lastValue = (0, _lodash.last)(list).value;
  let next = Number(lastValue) ? Number(lastValue) + 1 : 1;
  if (next >= range.max) {
    next = range.max - 1;
  }
  return {
    id: generateId(),
    value: next,
    isInvalid: false
  };
}
function getInitModelList(list) {
  return list.length ? list.map(num => ({
    value: num === undefined ? EMPTY_STRING : num,
    id: generateId(),
    isInvalid: false
  })) : [defaultModel];
}
function getValidatedModels(numberList, modelList, numberRange, validateAscendingOrder = false, disallowDuplicates = false) {
  if (!numberList.length) {
    return [defaultModel];
  }
  return numberList.map((number, index) => {
    const model = modelList[index] || {
      id: generateId()
    };
    const newValue = number === undefined ? EMPTY_STRING : number;
    const valueResult = numberRange ? validateValue(newValue, numberRange) : {
      isInvalid: false
    };
    const ascendingResult = validateAscendingOrder ? validateValueAscending(newValue, index, numberList) : {
      isInvalidOrder: false
    };
    const duplicationResult = disallowDuplicates ? validateValueUnique(newValue, index, numberList) : {
      isDuplicate: false
    };
    const allErrors = [valueResult.error, ascendingResult.error, duplicationResult.error].filter(Boolean).join(' ');
    return {
      ...model,
      value: newValue,
      isInvalid: valueResult.isInvalid || ascendingResult.isInvalidOrder || duplicationResult.isDuplicate,
      error: allErrors === EMPTY_STRING ? undefined : allErrors
    };
  });
}
function hasInvalidValues(modelList) {
  return !!modelList.find(({
    isInvalid
  }) => isInvalid);
}