"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchScreenshotBlocks = fetchScreenshotBlocks;
exports.generateBlockStatsOnPut = generateBlockStatsOnPut;
exports.pruneBlockCache = pruneBlockCache;
var _effects = require("redux-saga/effects");
var _runtime_types = require("../../../../common/runtime_types");
var _journey = require("../api/journey");
var _synthetics = require("../reducers/synthetics");
var _selectors = require("../selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function* fetchBlocks(hashes) {
  yield (0, _effects.put)((0, _synthetics.setBlockLoadingAction)(hashes));
  const blocks = yield (0, _effects.call)(_journey.fetchScreenshotBlockSet, hashes);
  yield (0, _effects.put)((0, _synthetics.putBlocksAction)({
    blocks
  }));
}
function* fetchScreenshotBlocks() {
  /**
   * We maintain a list of each hash and how many times it is requested so we can avoid
   * subsequent re-requests if the block is dropped due to cache pruning.
   */
  yield (0, _effects.takeEvery)(String(_synthetics.fetchBlocksAction), function* (action) {
    if (action.payload.length > 0) {
      yield (0, _effects.put)((0, _synthetics.updateHitCountsAction)(action.payload));
    }
  });

  /**
   * We do a short delay to allow multiple item renders to queue up before dispatching
   * a fetch to the backend.
   */
  yield (0, _effects.throttle)(20, String(_synthetics.fetchBlocksAction), function* () {
    const {
      blocks
    } = yield (0, _effects.select)(_selectors.syntheticsSelector);
    const toFetch = Object.keys(blocks).filter(hash => {
      const block = blocks[hash];
      return (0, _runtime_types.isPendingBlock)(block) && block.status !== 'loading';
    });
    if (toFetch.length > 0) {
      yield (0, _effects.fork)(fetchBlocks, toFetch);
    }
  });
}
function* generateBlockStatsOnPut() {
  yield (0, _effects.takeEvery)(String(_synthetics.putBlocksAction), function* (action) {
    const batchSize = action.payload.blocks.reduce((total, cur) => {
      return cur.synthetics.blob.length + total;
    }, 0);
    yield (0, _effects.put)((0, _synthetics.putCacheSize)(batchSize));
  });
}

// 4 MB cap for cache size
const MAX_CACHE_SIZE = 4000000;
function* pruneBlockCache() {
  yield (0, _effects.takeEvery)(String(_synthetics.putCacheSize), function* (_action) {
    const {
      cacheSize
    } = yield (0, _effects.select)(_selectors.syntheticsSelector);
    if (cacheSize > MAX_CACHE_SIZE) {
      yield (0, _effects.put)((0, _synthetics.pruneCacheAction)(cacheSize - MAX_CACHE_SIZE));
    }
  });
}