"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dataStreamMigrationServiceFactory = void 0;
var _rxjs = require("rxjs");
var _lodash = _interopRequireDefault(require("lodash"));
var _types = require("../../../common/types");
var _error = require("./error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dataStreamMigrationServiceFactory = ({
  esClient,
  licensing
}) => {
  return {
    hasRequiredPrivileges: async dataStreamName => {
      /**
       * To avoid a circular dependency on Security we use a work around
       * here to detect whether Security is available and enabled
       * (i.e., via the licensing plugin). This enables Security to use
       * functionality exposed through Upgrade Assistant.
       */
      const license = await (0, _rxjs.firstValueFrom)(licensing.license$);
      const securityFeature = license.getFeature('security');

      // If security is disabled or unavailable, return true.
      if (!securityFeature || !(securityFeature.isAvailable && securityFeature.isEnabled)) {
        return true;
      }
      const names = [dataStreamName];
      const resp = await esClient.security.hasPrivileges({
        body: {
          cluster: ['manage', 'cancel_task'],
          index: [{
            names,
            allow_restricted_indices: true,
            privileges: ['all']
          }]
        }
      });
      return resp.has_all_requested;
    },
    async detectMigrationWarnings() {
      return [{
        warningType: 'affectExistingSetups',
        resolutionType: 'readonly'
      }, {
        warningType: 'incompatibleDataStream',
        resolutionType: 'reindex'
      }];
    },
    async createReindexOperation(dataStreamName) {
      const indexExists = await esClient.indices.exists({
        index: dataStreamName
      });
      if (!indexExists) {
        throw _error.error.indexNotFound(`Index ${dataStreamName} does not exist in this cluster.`);
      }
      try {
        const result = await esClient.transport.request({
          method: 'POST',
          path: '/_migration/reindex',
          body: {
            mode: 'upgrade',
            source: {
              index: dataStreamName
            }
          }
        });
        if (!result.acknowledged) {
          throw _error.error.reindexTaskFailed(`The reindex operation failed to start for ${dataStreamName}`);
        }
        return true;
      } catch (err) {
        if (err.status === 400 && err.error.type === 'resource_already_exists_exception') {
          throw _error.error.reindexAlreadyInProgress(`A reindex operation already in-progress for ${dataStreamName}`);
        }
        throw _error.error.reindexTaskFailed(`The reindex operation failed to start for ${dataStreamName}`);
      }
    },
    async fetchMigrationStatus(dataStreamName) {
      // Check reindexing task progress
      try {
        const taskResponse = await esClient.transport.request({
          method: 'GET',
          path: `/_migration/reindex/${dataStreamName}/_status`
        });
        if (taskResponse.exception) {
          // Include the entire task result in the error message. This should be guaranteed
          // to be JSON-serializable since it just came back from Elasticsearch.
          throw _error.error.reindexTaskFailed(`Data Stream Reindexing exception:\n${taskResponse.exception}\n${JSON.stringify(taskResponse, null, 2)}`);
        }

        // Propagate errors from the reindex task even if reindexing is not yet complete.
        if (taskResponse.errors.length) {
          // Include the entire task result in the error message. This should be guaranteed
          // to be JSON-serializable since it just came back from Elasticsearch.
          throw _error.error.reindexTaskFailed(`Reindexing failed with ${taskResponse.errors.length} errors:\n${JSON.stringify(taskResponse, null, 2)}`);
        }
        if (taskResponse.complete) {
          var _taskResponse$in_prog, _taskResponse$errors;
          /**
           * If the task is complete, check if there are any remaining indices that require upgrade
           * If that is the case, we need to update the status to not started
           * This way the user can trigger a new migration.
           * Note: This is the best place to do this call because we it'll only be called
           * 1 timeonce the task is complete.
           * Cases we reach this code execution:
           *     1. Task is complete and the user has the UA open. It'll disappear once the user refreshes.
           *     2. Task is complete but we have remaining indices that require upgrade.
           */

          const {
            data_streams: dataStreamsDeprecations
          } = await esClient.migration.deprecations({
            filter_path: `data_streams`
          });
          const deprecationsDetails = dataStreamsDeprecations[dataStreamName];
          if (deprecationsDetails && deprecationsDetails.length) {
            const deprecationDetails = deprecationsDetails.find(deprecation => deprecation._meta.reindex_required);
            if (deprecationDetails) {
              const stillNeedsUpgrade = deprecationDetails._meta.reindex_required === true && deprecationDetails._meta.indices_requiring_upgrade_count > 0;
              if (stillNeedsUpgrade) {
                return {
                  status: _types.DataStreamMigrationStatus.notStarted
                };
              }
            }
          }

          // Find the first deprecation that has reindex_required set to true
          // Update the status
          return {
            taskPercComplete: 1,
            status: _types.DataStreamMigrationStatus.completed,
            resolutionType: 'reindex',
            progressDetails: {
              startTimeMs: taskResponse.start_time_millis,
              successCount: taskResponse.successes,
              pendingCount: taskResponse.pending,
              inProgressCount: ((_taskResponse$in_prog = taskResponse.in_progress) !== null && _taskResponse$in_prog !== void 0 ? _taskResponse$in_prog : []).length,
              errorsCount: ((_taskResponse$errors = taskResponse.errors) !== null && _taskResponse$errors !== void 0 ? _taskResponse$errors : []).length
            }
          };
        } else {
          var _taskResponse$in_prog2, _taskResponse$errors2;
          // Updated the percent complete
          const perc = taskResponse.successes / taskResponse.total_indices_in_data_stream;
          return {
            status: _types.DataStreamMigrationStatus.inProgress,
            taskPercComplete: perc,
            resolutionType: 'reindex',
            progressDetails: {
              startTimeMs: taskResponse.start_time_millis,
              successCount: taskResponse.successes,
              pendingCount: taskResponse.pending,
              inProgressCount: ((_taskResponse$in_prog2 = taskResponse.in_progress) !== null && _taskResponse$in_prog2 !== void 0 ? _taskResponse$in_prog2 : []).length,
              errorsCount: ((_taskResponse$errors2 = taskResponse.errors) !== null && _taskResponse$errors2 !== void 0 ? _taskResponse$errors2 : []).length
            }
          };
        }
      } catch (err) {
        if (err.name === 'ResponseError' && err.message.includes('resource_not_found_exception')) {
          // cancelled, never started, or successful task but finished from than 24 hours ago
          // Since this API should be called as a follow up from _migrate API, we can assume that the task is not started
          return {
            status: _types.DataStreamMigrationStatus.notStarted
          };
        }
        return {
          status: _types.DataStreamMigrationStatus.failed,
          resolutionType: 'reindex',
          errorMessage: err.toString()
        };
      }
    },
    async cancelReindexing(dataStreamName) {
      const resp = await esClient.transport.request({
        method: 'POST',
        path: `/_migration/reindex/${dataStreamName}/_cancel`
      });
      if (!resp.acknowledged) {
        throw _error.error.reindexCannotBeCancelled(`Could not cancel reindex.`);
      }
      return {
        status: _types.DataStreamMigrationStatus.cancelled,
        resolutionType: 'reindex'
      };
    },
    async getDataStreamMetadata(dataStreamName) {
      try {
        const {
          body: statsBody
        } = await esClient.transport.request({
          method: 'GET',
          path: `/${dataStreamName}/_stats`
        }, {
          meta: true
        });
        const {
          data_streams: dataStreamsDeprecations
        } = await esClient.migration.deprecations({
          filter_path: `data_streams`
        });
        const deprecationsDetails = dataStreamsDeprecations[dataStreamName];
        if (!deprecationsDetails || !deprecationsDetails.length) {
          return null;
        }

        // Find the first deprecation that has reindex_required set to true
        const deprecationDetails = deprecationsDetails.find(deprecation => deprecation._meta.reindex_required);
        if (!deprecationDetails) {
          return null;
        }
        const indicesRequiringUpgrade = deprecationDetails._meta.indices_requiring_upgrade;
        const allIndices = Object.keys(statsBody.indices);
        let indicesRequiringUpgradeDocsCount = 0;
        let indicesRequiringUpgradeDocsSize = 0;
        let oldestIncompatibleDocTimestamp;
        const indicesCreationDates = [];
        for (const index of indicesRequiringUpgrade) {
          const indexStats = Object.entries(statsBody.indices).find(([key]) => key === index);
          if (!indexStats) {
            throw _error.error.cannotGrabMetadata(`Index ${index} does not exist in this cluster.`);
          }
          indicesRequiringUpgradeDocsSize += indexStats[1].primaries.store.total_data_set_size_in_bytes;
          indicesRequiringUpgradeDocsCount += indexStats[1].primaries.docs.count;
          const body = await esClient.indices.getSettings({
            index,
            flat_settings: true
          });
          const creationDate = _lodash.default.get(body, [index, 'settings', 'index.creation_date']);
          if (creationDate) {
            indicesCreationDates.push(creationDate);
          }

          // Check if the index has documents before checking for oldest timestamp
          if (indexStats[1].primaries.docs.count > 0) {
            try {
              var _timestampResponse$ag, _timestampResponse$ag2;
              // Find the oldest document timestamp in incompatible index
              const timestampResponse = await esClient.search({
                index,
                size: 0,
                body: {
                  aggs: {
                    oldest_incompatible_doc: {
                      min: {
                        field: '@timestamp'
                      }
                    }
                  }
                }
              });
              const oldestTimestamp = // @ts-ignore - value doesnt exist in the es type yet
              (_timestampResponse$ag = timestampResponse.aggregations) === null || _timestampResponse$ag === void 0 ? void 0 : (_timestampResponse$ag2 = _timestampResponse$ag.oldest_incompatible_doc) === null || _timestampResponse$ag2 === void 0 ? void 0 : _timestampResponse$ag2.value;
              if (oldestTimestamp && (!oldestIncompatibleDocTimestamp || oldestTimestamp < oldestIncompatibleDocTimestamp)) {
                oldestIncompatibleDocTimestamp = oldestTimestamp;
              }
            } catch (err) {
              // If aggregation fails (possibly due to missing @timestamp field), continue without setting timestamp
              // This prevents a single index without proper timestamps from breaking the entire metadata function
            }
          }
        }
        const lastIndexRequiringUpgradeCreationDate = Math.max(...indicesCreationDates);
        return {
          dataStreamName,
          documentationUrl: deprecationDetails.url,
          allIndices,
          allIndicesCount: allIndices.length,
          indicesRequiringUpgrade,
          indicesRequiringUpgradeCount: indicesRequiringUpgrade.length,
          lastIndexRequiringUpgradeCreationDate,
          indicesRequiringUpgradeDocsSize,
          indicesRequiringUpgradeDocsCount,
          oldestIncompatibleDocTimestamp
        };
      } catch (err) {
        throw _error.error.cannotGrabMetadata(`Could not grab metadata for ${dataStreamName}. ${err.message.toString()}`);
      }
    },
    async readonlyIndices(dataStreamName, indices) {
      try {
        const {
          data_streams: dataStreamsDetails
        } = await esClient.indices.getDataStream({
          name: dataStreamName
        });
        // Since we are not using a pattern it should only return one item
        const dataStreamBackIndices = dataStreamsDetails[0].indices;

        // The last item in this array contains information about the stream’s current write index.
        const writeIndex = dataStreamBackIndices[dataStreamBackIndices.length - 1].index_name;
        const hasWriteIndex = indices.some(index => index === writeIndex);
        if (hasWriteIndex) {
          const rollOverResponse = await esClient.indices.rollover({
            alias: dataStreamName
          });
          if (!rollOverResponse.acknowledged) {
            throw _error.error.readonlyTaskFailed(`Could not rollover data stream ${dataStreamName}.`);
          }
        }
      } catch (err) {
        throw _error.error.readonlyTaskFailed(`Could not migrate data stream ${dataStreamName}.`);
      }
      for (const index of indices) {
        try {
          const unfreeze = await esClient.indices.unfreeze({
            index
          });
          const addBlock = await esClient.indices.addBlock({
            index,
            block: 'write'
          });
          if (!unfreeze.acknowledged || !addBlock.acknowledged) {
            throw _error.error.readonlyTaskFailed(`Could not set index ${index} to readonly.`);
          }
        } catch (err) {
          if (err instanceof _error.DataStreamMigrationError) {
            throw err;
          }
          // ES errors are serializable, so we can just stringify the error and throw it.
          const stringifiedErr = JSON.stringify(err, null, 2);
          throw _error.error.readonlyTaskFailed(`Could not migrate index "${index}". Got: ${stringifiedErr}`);
        }
      }
    }
  };
};
exports.dataStreamMigrationServiceFactory = dataStreamMigrationServiceFactory;