"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.apiService = exports.ApiService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("../../../common/constants");
var _shared_imports = require("../../shared_imports");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ApiService {
  constructor() {
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "clusterUpgradeStateListeners", []);
  }
  handleClusterUpgradeError(error) {
    const isClusterUpgradeError = Boolean(error && error.statusCode === 426);
    if (isClusterUpgradeError) {
      const clusterUpgradeState = error.attributes.allNodesUpgraded ? 'isUpgradeComplete' : 'isUpgrading';
      this.clusterUpgradeStateListeners.forEach(listener => listener(clusterUpgradeState));
    }
  }
  useRequest(config) {
    if (!this.client) {
      throw new Error('API service has not been initialized.');
    }
    const response = (0, _shared_imports.useRequest)(this.client, config);
    // NOTE: This will cause an infinite render loop in any component that both
    // consumes the hook calling this useRequest function and also handles
    // cluster upgrade errors. Note that sendRequest doesn't have this problem.
    //
    // This is due to React's fundamental expectation that hooks be idempotent,
    // so it can render a component as many times as necessary and thereby call
    // the hook on each render without worrying about that triggering subsequent
    // renders.
    //
    // In this case we call handleClusterUpgradeError every time useRequest is
    // called, which is on every render. If handling the cluster upgrade error
    // causes a state change in the consuming component, that will trigger a
    // render, which will call useRequest again, calling handleClusterUpgradeError,
    // causing a state change in the consuming component, and so on.
    this.handleClusterUpgradeError(response.error);
    return response;
  }
  async sendRequest(config) {
    if (!this.client) {
      throw new Error('API service has not been initialized.');
    }
    const response = await (0, _shared_imports.sendRequest)(this.client, config);
    this.handleClusterUpgradeError(response.error);
    return response;
  }
  setup(httpClient) {
    this.client = httpClient;
  }
  onClusterUpgradeStateChange(listener) {
    this.clusterUpgradeStateListeners.push(listener);
  }
  useLoadClusterUpgradeStatus() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/cluster_upgrade_status`,
      method: 'get',
      pollIntervalMs: _constants.CLUSTER_UPGRADE_STATUS_POLL_INTERVAL_MS
    });
  }
  useLoadCloudBackupStatus() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/cloud_backup_status`,
      method: 'get',
      pollIntervalMs: _constants.CLOUD_BACKUP_STATUS_POLL_INTERVAL_MS
    });
  }
  useLoadSystemIndicesMigrationStatus() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/system_indices_migration`,
      method: 'get'
    });
  }
  async migrateSystemIndices() {
    const result = await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/system_indices_migration`,
      method: 'post'
    });
    return result;
  }
  useLoadEsDeprecations() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/es_deprecations`,
      method: 'get'
    });
  }
  useLoadDeprecationLogging() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/deprecation_logging`,
      method: 'get'
    });
  }
  async updateDeprecationLogging(loggingData) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/deprecation_logging`,
      method: 'put',
      body: JSON.stringify(loggingData)
    });
  }
  getDeprecationLogsCount(from) {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/deprecation_logging/count`,
      method: 'get',
      query: {
        from
      },
      pollIntervalMs: _constants.DEPRECATION_LOGS_COUNT_POLL_INTERVAL_MS
    });
  }
  deleteDeprecationLogsCache() {
    return this.sendRequest({
      path: `${_constants.API_BASE_PATH}/deprecation_logging/cache`,
      method: 'delete'
    });
  }
  async updateIndexSettings(indexName, settings) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/${indexName}/index_settings`,
      method: 'post',
      body: {
        settings: JSON.stringify(settings)
      }
    });
  }
  async upgradeMlSnapshot(body) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/ml_snapshots`,
      method: 'post',
      body
    });
  }
  async deleteMlSnapshot({
    jobId,
    snapshotId
  }) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/ml_snapshots/${jobId}/${snapshotId}`,
      method: 'delete'
    });
  }
  async getMlSnapshotUpgradeStatus({
    jobId,
    snapshotId
  }) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/ml_snapshots/${jobId}/${snapshotId}`,
      method: 'get'
    });
  }
  useLoadMlUpgradeMode() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/ml_upgrade_mode`,
      method: 'get'
    });
  }

  /**
   * Data Stream Migrations
   * Reindex and readonly operations
   */

  async getDataStreamMigrationStatus(dataStreamName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/migrate_data_stream/${dataStreamName}`,
      method: 'get'
    });
  }
  async getDataStreamMetadata(dataStreamName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/migrate_data_stream/${dataStreamName}/metadata`,
      method: 'get'
    });
  }
  async startDataStreamReindexTask(dataStreamName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/migrate_data_stream/${dataStreamName}/reindex`,
      method: 'post'
    });
  }
  async cancelDataStreamReindexTask(dataStreamName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/migrate_data_stream/${dataStreamName}/reindex/cancel`,
      method: 'post'
    });
  }
  async markIndicesAsReadOnly(dataStreamName, indices) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/migrate_data_stream/${dataStreamName}/readonly`,
      method: 'post',
      body: {
        indices
      }
    });
  }
  async deleteDataStream(dataStreamName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/migrate_data_stream/${dataStreamName}`,
      method: 'delete'
    });
  }
  /**
   * FINISH: Data Stream Migrations
   */

  async getReindexStatus(indexName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/reindex/${indexName}`,
      method: 'get'
    });
  }
  async startReindexTask(indexName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/reindex/${indexName}`,
      method: 'post'
    });
  }
  async cancelReindexTask(indexName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/reindex/${indexName}/cancel`,
      method: 'post'
    });
  }
  async updateIndex(indexName, operations) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/update_index/${indexName}`,
      method: 'post',
      body: {
        operations
      }
    });
  }
  async deleteIndex(indexName) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/delete_index/${indexName}`,
      method: 'delete'
    });
  }
  useLoadUpgradeStatus() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/status`,
      method: 'get',
      pollIntervalMs: _constants.UPGRADE_STATUS_POLL_INTERVAL_MS
    });
  }
  async updateClusterSettings(settings) {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/cluster_settings`,
      method: 'post',
      body: {
        settings: JSON.stringify(settings)
      }
    });
  }
  async getMLEnabled() {
    return await this.sendRequest({
      path: `${_constants.API_BASE_PATH}/ml_enabled`,
      method: 'get'
    });
  }
  useLoadRemoteClusters() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/remote_clusters`,
      method: 'get'
    });
  }
  useLoadNodeDiskSpace() {
    return this.useRequest({
      path: `${_constants.API_BASE_PATH}/node_disk_space`,
      method: 'get'
    });
  }
}
exports.ApiService = ApiService;
const apiService = exports.apiService = new ApiService();