"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SyntheticsMonitorClient = exports.LIGHTWEIGHT_TEST_NOW_RUN = exports.BROWSER_TEST_NOW_RUN = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("../utils");
var _synthetics_private_location = require("../private_location/synthetics_private_location");
var _runtime_types = require("../../../common/runtime_types");
var _format_configs = require("../formatters/public_formatters/format_configs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LIGHTWEIGHT_TEST_NOW_RUN = exports.LIGHTWEIGHT_TEST_NOW_RUN = 'LIGHTWEIGHT_SYNTHETICS_TEST_NOW_RUN';
const BROWSER_TEST_NOW_RUN = exports.BROWSER_TEST_NOW_RUN = 'BROWSER_SYNTHETICS_TEST_NOW_RUN';
const LONG_TIME_MONTH = '43800';
class SyntheticsMonitorClient {
  constructor(syntheticsService, server) {
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "syntheticsService", void 0);
    (0, _defineProperty2.default)(this, "privateLocationAPI", void 0);
    this.server = server;
    this.syntheticsService = syntheticsService;
    this.privateLocationAPI = new _synthetics_private_location.SyntheticsPrivateLocation(server);
  }
  async addMonitors(monitors, allPrivateLocations, spaceId) {
    const privateConfigs = [];
    const publicConfigs = [];
    const paramsBySpace = await this.syntheticsService.getSyntheticsParams({
      spaceId
    });
    const maintenanceWindows = await this.syntheticsService.getMaintenanceWindows();
    for (const monitorObj of monitors) {
      const {
        formattedConfig,
        params,
        config
      } = await this.formatConfigWithParams(monitorObj, spaceId, paramsBySpace);
      const {
        privateLocations,
        publicLocations
      } = this.parseLocations(formattedConfig);
      if (privateLocations.length > 0) {
        privateConfigs.push({
          config: formattedConfig,
          globalParams: params
        });
      }
      if (publicLocations.length > 0) {
        publicConfigs.push(config);
      }
    }
    const newPolicies = this.privateLocationAPI.createPackagePolicies(privateConfigs, allPrivateLocations, spaceId, maintenanceWindows);
    const syncErrors = this.syntheticsService.addConfigs(publicConfigs, maintenanceWindows);
    return await Promise.all([newPolicies, syncErrors]);
  }
  async editMonitors(monitors, allPrivateLocations, spaceId) {
    const privateConfigs = [];
    const publicConfigs = [];
    const deletedPublicConfigs = [];
    const paramsBySpace = await this.syntheticsService.getSyntheticsParams({
      spaceId
    });
    const maintenanceWindows = await this.syntheticsService.getMaintenanceWindows();
    for (const editedMonitor of monitors) {
      const {
        str: paramsString,
        params
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], editedMonitor.monitor);
      const configData = {
        spaceId,
        params: paramsBySpace[spaceId],
        monitor: editedMonitor.monitor,
        configId: editedMonitor.id
      };
      const editedConfig = (0, _format_configs.formatHeartbeatRequest)(configData, paramsString);
      const {
        publicLocations,
        privateLocations
      } = this.parseLocations(editedConfig);
      if (publicLocations.length > 0) {
        publicConfigs.push(configData);
      }
      const deletedPublicConfig = this.hasDeletedPublicLocations(publicLocations, editedMonitor.decryptedPreviousMonitor);
      if (deletedPublicConfig) {
        deletedPublicConfigs.push({
          ...deletedPublicConfig,
          params: paramsBySpace[spaceId],
          spaceId
        });
      }
      if (privateLocations.length > 0 || this.hasPrivateLocations(editedMonitor.decryptedPreviousMonitor)) {
        privateConfigs.push({
          config: editedConfig,
          globalParams: params
        });
      }
    }
    if (deletedPublicConfigs.length > 0) {
      await this.syntheticsService.deleteConfigs(deletedPublicConfigs);
    }
    const privateEditPromise = this.privateLocationAPI.editMonitors(privateConfigs, allPrivateLocations, spaceId, maintenanceWindows);
    const publicConfigsPromise = this.syntheticsService.editConfig(publicConfigs, true, maintenanceWindows);
    const [publicSyncErrors, privateEditResponse] = await Promise.all([publicConfigsPromise, privateEditPromise]);
    const {
      failedUpdates: failedPolicyUpdates
    } = privateEditResponse;
    return {
      failedPolicyUpdates,
      publicSyncErrors
    };
  }
  async deleteMonitors(monitors, spaceId) {
    const privateDeletePromise = this.privateLocationAPI.deleteMonitors(monitors, spaceId);
    const publicDeletePromise = this.syntheticsService.deleteConfigs(monitors.map(monitor => ({
      spaceId,
      monitor,
      configId: monitor.config_id,
      params: {}
    })));
    const [pubicResponse] = await Promise.all([publicDeletePromise, privateDeletePromise]);
    return pubicResponse;
  }
  async testNowConfigs(monitor, savedObjectsClient, allPrivateLocations, spaceId, runOnce) {
    let privateConfig;
    let publicConfig;
    const paramsBySpace = await this.syntheticsService.getSyntheticsParams({
      spaceId
    });
    const {
      formattedConfig,
      params,
      config
    } = await this.formatConfigWithParams(monitor, spaceId, paramsBySpace);
    const {
      privateLocations,
      publicLocations
    } = this.parseLocations(formattedConfig);
    if (privateLocations.length > 0) {
      privateConfig = {
        config: {
          ...formattedConfig,
          [_runtime_types.ConfigKey.SCHEDULE]: {
            number: LONG_TIME_MONTH,
            unit: _runtime_types.ScheduleUnit.MINUTES
          },
          [_runtime_types.ConfigKey.ENABLED]: true
        },
        globalParams: params
      };
    }
    if (publicLocations.length > 0) {
      publicConfig = config;
      // making it enabled, even if it's disabled in the UI
      publicConfig.monitor.enabled = true;
      publicConfig.testRunId = monitor.testRunId;
      if (runOnce) {
        publicConfig.runOnce = true;
      }
    }
    const newPolicies = this.privateLocationAPI.createPackagePolicies(privateConfig ? [privateConfig] : [], allPrivateLocations, spaceId, [], monitor.testRunId, runOnce);
    const syncErrors = this.syntheticsService.runOnceConfigs(publicConfig);
    return await Promise.all([newPolicies, syncErrors]);
  }
  hasPrivateLocations(previousMonitor) {
    const {
      locations
    } = previousMonitor.attributes;
    return locations.some(loc => !loc.isServiceManaged);
  }
  hasDeletedPublicLocations(updatedLocations, decryptedPreviousMonitor) {
    const {
      locations
    } = decryptedPreviousMonitor.attributes;
    const prevPublicLocations = locations.filter(loc => loc.isServiceManaged);
    const missingPublicLocations = prevPublicLocations.filter(prevLoc => {
      return !updatedLocations.some(updatedLoc => updatedLoc.id === prevLoc.id);
    });
    if (missingPublicLocations.length > 0) {
      const {
        attributes: normalizedPreviousMonitor
      } = (0, _utils.normalizeSecrets)(decryptedPreviousMonitor);
      normalizedPreviousMonitor.locations = missingPublicLocations;
      return {
        monitor: normalizedPreviousMonitor,
        configId: decryptedPreviousMonitor.id
      };
    }
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  async formatConfigWithParams(monitorObj, spaceId, paramsBySpace) {
    const {
      monitor,
      id
    } = monitorObj;
    const config = {
      spaceId,
      monitor,
      configId: id,
      params: paramsBySpace[spaceId]
    };
    const {
      str: paramsString,
      params
    } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], monitor);
    const formattedConfig = (0, _format_configs.formatHeartbeatRequest)(config, paramsString);
    return {
      formattedConfig,
      params,
      config
    };
  }
  async inspectMonitor(monitorObj, allPrivateLocations, spaceId, hideParams, canSave) {
    const privateConfigs = [];
    const paramsBySpace = await this.syntheticsService.getSyntheticsParams({
      spaceId,
      canSave,
      hideParams
    });
    const maintenanceWindows = await this.syntheticsService.getMaintenanceWindows();
    const {
      formattedConfig,
      params,
      config
    } = await this.formatConfigWithParams(monitorObj, spaceId, paramsBySpace);
    if (hideParams) {
      formattedConfig.params = hideParamsHelper(formattedConfig.params);
      config.monitor.params = hideParamsHelper(config.monitor.params);
    }
    const {
      privateLocations,
      publicLocations
    } = this.parseLocations(formattedConfig);
    if (privateLocations.length > 0) {
      privateConfigs.push({
        config: formattedConfig,
        globalParams: params
      });
    }
    const publicPromise = this.syntheticsService.inspectConfig(publicLocations.length > 0 ? config : null, maintenanceWindows);
    const privatePromise = this.privateLocationAPI.inspectPackagePolicy({
      privateConfig: privateConfigs === null || privateConfigs === void 0 ? void 0 : privateConfigs[0],
      allPrivateLocations,
      maintenanceWindows,
      spaceId
    });
    const [publicConfigs, privateConfig] = await Promise.all([publicPromise, privatePromise]);
    return {
      publicConfigs,
      privateConfig
    };
  }
}
exports.SyntheticsMonitorClient = SyntheticsMonitorClient;
const hideParamsHelper = params => {
  if (!params) return params;
  const parsedParams = JSON.parse(params);
  // replace all values with '***'
  const newParams = Object.create(null);
  Object.keys(parsedParams).forEach(key => {
    newParams[key] = '"********"';
  });
  return JSON.stringify(newParams);
};