"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateArchive = generateArchive;
exports.parseArchive = parseArchive;
var _yaml = _interopRequireDefault(require("yaml"));
var _contentPacksSchema = require("@kbn/content-packs-schema");
var _admZip = _interopRequireDefault(require("adm-zip"));
var _path = _interopRequireDefault(require("path"));
var _lodash = require("lodash");
var _error = require("./error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ARCHIVE_ENTRY_MAX_SIZE_BYTES = 1 * 1024 * 1024; // 1MB

async function parseArchive(archive) {
  const zip = await new Promise((resolve, reject) => {
    const bufs = [];
    archive.on('data', chunk => bufs.push(chunk));
    archive.on('end', () => {
      try {
        resolve(new _admZip.default(Buffer.concat(bufs)));
      } catch (err) {
        reject(new _error.InvalidContentPackError('Invalid content pack format'));
      }
    });
    archive.on('error', error => reject(error));
  });
  const rootDir = getRootDir(zip.getEntries());
  const manifest = await extractManifest(rootDir, zip);
  const entries = await extractEntries(rootDir, zip);
  return {
    ...manifest,
    entries
  };
}
async function generateArchive(manifest, objects) {
  const zip = new _admZip.default();
  const rootDir = `${manifest.name}-${manifest.version}`;
  objects.filter(object => (0, _contentPacksSchema.isSupportedEntryType)(object.type)).forEach(object => {
    const type = object.type;
    switch (type) {
      case 'dashboard':
      case 'index-pattern':
      case 'lens':
        const subDir = _contentPacksSchema.SUPPORTED_SAVED_OBJECT_TYPE[object.type];
        zip.addFile(_path.default.join(rootDir, 'kibana', subDir, `${object.id}.json`), Buffer.from(JSON.stringify(object, null, 2)));
        return;
      default:
        missingEntryTypeImpl(type);
    }
  });
  zip.addFile(_path.default.join(rootDir, 'manifest.yml'), Buffer.from(_yaml.default.stringify((0, _lodash.pick)(manifest, ['name', 'description', 'version']))));
  return zip.toBufferPromise();
}
async function readEntry(entry) {
  const buf = await new Promise((resolve, reject) => {
    entry.getDataAsync((data, err) => {
      if (err) return reject(new Error(err));
      resolve(data);
    });
  });
  return buf;
}
async function extractManifest(rootDir, zip) {
  const manifestPath = `${rootDir}/manifest.yml`;
  const entry = zip.getEntry(manifestPath);
  if (!entry) {
    throw new _error.InvalidContentPackError(`Expected manifest at [${manifestPath}]`);
  }
  assertUncompressedSize(entry);
  const {
    data: manifest,
    success
  } = _contentPacksSchema.contentPackManifestSchema.safeParse(_yaml.default.parse((await readEntry(entry)).toString()));
  if (!success) {
    throw new _error.InvalidContentPackError('Invalid content pack manifest format');
  }
  return manifest;
}
async function extractEntries(rootDir, zip) {
  const supportedEntries = zip.getEntries().filter(entry => (0, _contentPacksSchema.isSupportedFile)(rootDir, entry.entryName));
  supportedEntries.forEach(entry => assertUncompressedSize(entry));
  const entries = await Promise.all(supportedEntries.map(entry => {
    const type = (0, _contentPacksSchema.getEntryTypeByFile)(rootDir, entry.entryName);
    switch (type) {
      case 'lens':
      case 'index-pattern':
        // these are handled by their parent dashboard
        return [];
      case 'dashboard':
        return resolveDashboard(rootDir, zip, entry);
      default:
        missingEntryTypeImpl(type);
    }
  }));
  return entries.flat();
}
async function resolveDashboard(rootDir, zip, dashboardEntry) {
  const dashboard = JSON.parse((await readEntry(dashboardEntry)).toString());
  const uniqReferences = (0, _lodash.uniqBy)(dashboard.references, ref => ref.id);
  if (uniqReferences.some(({
    type
  }) => !(0, _contentPacksSchema.isSupportedReferenceType)(type))) {
    throw new _error.InvalidContentPackError(`Dashboard [${dashboard.id}] references saved object types not supported by content packs: ${uniqReferences.filter(({
      type
    }) => !(0, _contentPacksSchema.isSupportedReferenceType)(type))}`);
  }
  const includedReferences = (0, _lodash.compact)(uniqReferences.map(ref => zip.getEntry(_path.default.join(rootDir, 'kibana', _contentPacksSchema.SUPPORTED_SAVED_OBJECT_TYPE[ref.type], `${ref.id}.json`))));
  const resolvedReferences = await Promise.all(includedReferences.map(async entry => JSON.parse((await readEntry(entry)).toString())));
  return [dashboard, ...resolvedReferences];
}
function getRootDir(entries) {
  const rootDirs = new Set();
  for (const entry of entries) {
    const rootDir = entry.entryName.split(_path.default.sep)[0];
    rootDirs.add(rootDir);
  }
  if (rootDirs.size !== 1) {
    throw new _error.InvalidContentPackError(`Expected a single root directory but got [${Array.from(rootDirs)}]`);
  }
  return rootDirs.keys().next().value;
}
function assertUncompressedSize(entry) {
  if (entry.header.size > ARCHIVE_ENTRY_MAX_SIZE_BYTES) {
    throw new _error.InvalidContentPackError(`Object [${entry.entryName}] exceeds the limit of ${ARCHIVE_ENTRY_MAX_SIZE_BYTES} bytes`);
  }
}
function missingEntryTypeImpl(type) {
  throw new Error(`Content pack entry type [${type}] is not implemented`);
}