"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResetSLO = void 0;
var _std = require("@kbn/std");
var _sloSchema = require("@kbn/slo-schema");
var _constants = require("../../common/constants");
var _sli_pipeline_template = require("../assets/ingest_templates/sli_pipeline_template");
var _summary_pipeline_template = require("../assets/ingest_templates/summary_pipeline_template");
var _errors = require("../errors");
var _retry = require("../utils/retry");
var _create_temp_summary = require("./summary_transform_generator/helpers/create_temp_summary");
var _assert_expected_indicator_source_index_privileges = require("./utils/assert_expected_indicator_source_index_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ResetSLO {
  constructor(scopedClusterClient, repository, transformManager, summaryTransformManager, logger, spaceId, basePath) {
    this.scopedClusterClient = scopedClusterClient;
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.logger = logger;
    this.spaceId = spaceId;
    this.basePath = basePath;
  }
  async execute(sloId) {
    const originalSlo = await this.repository.findById(sloId);
    await (0, _assert_expected_indicator_source_index_privileges.assertExpectedIndicatorSourceIndexPrivileges)(originalSlo, this.scopedClusterClient.asCurrentUser);
    await this.deleteOriginalSLO(originalSlo);
    const resetedSlo = {
      ...originalSlo,
      revision: originalSlo.revision + 1,
      version: _constants.SLO_MODEL_VERSION,
      updatedAt: new Date()
    };
    await this.installResetedSLO(resetedSlo);
    await this.repository.update(resetedSlo);
    return _sloSchema.resetSLOResponseSchema.encode(resetedSlo);
  }
  async deleteOriginalSLO(slo) {
    try {
      await Promise.all([this.transformManager.uninstall((0, _constants.getSLOTransformId)(slo.id, slo.revision)), this.summaryTransformManager.uninstall((0, _constants.getSLOSummaryTransformId)(slo.id, slo.revision)), this.deletePipeline((0, _constants.getWildcardPipelineId)(slo.id, slo.revision))]);

      // Delete after we uninstalled the transforms
      await Promise.all([this.deleteRollupData(slo), this.deleteSummaryData(slo)]);
    } catch (err) {
      // Any errors here should not prevent moving forward.
      // Worst case we keep rolling up data for the orignal SLO
      this.logger.debug(`Deletion of the original SLO resources failed while resetting it: ${err}.`);
    }
  }
  async deleteRollupData(slo) {
    await this.scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SLI_DESTINATION_INDEX_PATTERN,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': slo.id
            }
          }, {
            term: {
              'slo.revision': slo.revision
            }
          }]
        }
      }
    });
  }
  async deleteSummaryData(slo) {
    await this.scopedClusterClient.asCurrentUser.deleteByQuery({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: true,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': slo.id
            }
          }, {
            term: {
              'slo.revision': slo.revision
            }
          }]
        }
      }
    });
  }
  async deletePipeline(id) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.deletePipeline({
      id
    }, {
      ignore: [404]
    }), {
      logger: this.logger
    });
  }
  async installResetedSLO(slo) {
    const rollbackOperations = [];
    const {
      id,
      revision
    } = slo;
    const rollupTransformId = (0, _constants.getSLOTransformId)(id, revision);
    const summaryTransformId = (0, _constants.getSLOSummaryTransformId)(id, revision);
    try {
      const sloPipelinePromise = this.createPipeline((0, _sli_pipeline_template.getSLIPipelineTemplate)(slo, this.spaceId));
      rollbackOperations.push(() => this.deletePipeline((0, _constants.getSLOPipelineId)(id, revision)));
      const rollupTransformPromise = this.transformManager.install(slo);
      rollbackOperations.push(() => this.transformManager.uninstall(rollupTransformId));
      const summaryPipelinePromise = this.createPipeline((0, _summary_pipeline_template.getSummaryPipelineTemplate)(slo, this.spaceId, this.basePath));
      rollbackOperations.push(() => this.deletePipeline((0, _constants.getSLOSummaryPipelineId)(id, revision)));
      const summaryTransformPromise = this.summaryTransformManager.install(slo);
      rollbackOperations.push(() => this.summaryTransformManager.uninstall(summaryTransformId));
      const tempDocPromise = this.createTempSummaryDocument(slo);
      rollbackOperations.push(() => this.deleteTempSummaryDocument(slo));
      await Promise.all([sloPipelinePromise, rollupTransformPromise, summaryPipelinePromise, summaryTransformPromise, tempDocPromise]);

      // transforms can only be started after the pipelines are created
      await Promise.all([this.transformManager.start(rollupTransformId), this.summaryTransformManager.start(summaryTransformId)]);
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      this.logger.debug(`Cannot reset the SLO [id: ${slo.id}, revision: ${slo.revision}]. Rolling back. ${err}`);
      await (0, _std.asyncForEach)(rollbackOperations.reverse(), async operation => {
        try {
          await operation();
        } catch (rollbackErr) {
          this.logger.debug(`Rollback operation failed. ${rollbackErr}`);
        }
      });
      if (((_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'security_exception') {
        throw new _errors.SecurityException(err.meta.body.error.reason);
      }
      throw err;
    }
  }
  async createPipeline(params) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.putPipeline(params), {
      logger: this.logger
    });
  }
  async createTempSummaryDocument(slo) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asCurrentUser.index({
      index: _constants.SUMMARY_TEMP_INDEX_NAME,
      id: `slo-${slo.id}`,
      document: (0, _create_temp_summary.createTempSummaryDocument)(slo, this.spaceId, this.basePath),
      refresh: true
    }), {
      logger: this.logger
    });
  }
  async deleteTempSummaryDocument(slo) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asCurrentUser.delete({
      index: _constants.SUMMARY_TEMP_INDEX_NAME,
      id: `slo-${slo.id}`,
      refresh: true
    }), {
      logger: this.logger
    });
  }
}
exports.ResetSLO = ResetSLO;