"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DIAGNOSTIC_QUERIES_ENCRYPT_VERSION = void 0;
exports.encryptDEKWithRSA = encryptDEKWithRSA;
exports.encryptField = encryptField;
exports.generateDEK = generateDEK;
var _crypto = require("crypto");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DIAGNOSTIC_QUERIES_ENCRYPT_VERSION = exports.DIAGNOSTIC_QUERIES_ENCRYPT_VERSION = 'v1';

/**
 * Represents an encrypted field
 */

/**
 * Generate random 256-bit Data Encryption Key (DEK)
 *
 * @returns 256-bit random key
 */
function generateDEK() {
  return (0, _crypto.randomBytes)(32);
}

/**
 * Encrypt DEK with RSA public key using OAEP padding
 *
 * @param dek - Data Encryption Key to encrypt
 * @param publicKeyPEM - RSA public key in PEM format
 * @returns Encrypted DEK (512 bytes for RSA-4096)
 */
function encryptDEKWithRSA(dek, publicKeyPEM) {
  return (0, _crypto.publicEncrypt)({
    key: publicKeyPEM,
    padding: _crypto.constants.RSA_PKCS1_OAEP_PADDING,
    oaepHash: 'sha256'
  }, dek);
}

/**
 * Encrypt a field value using AES-256-GCM and return formatted string
 *
 * Format: `CURRENT_VERSION:{keyId}:{encryptedDEK}:{iv}:{ciphertext}:{authTag}`
 *
 * @param value - Field value to encrypt
 * @param dek - Data Encryption Key (ephemeral, one per query execution)
 * @param encryptedDEK - DEK encrypted with RSA public key
 * @param keyId - Identifier for the RSA key pair used
 * @returns Formatted encrypted string
 */
function encryptField(value, dek, encryptedDEK, keyId) {
  const {
    iv,
    ciphertext,
    authTag
  } = encryptFieldWithAES(value, dek);
  const parts = [DIAGNOSTIC_QUERIES_ENCRYPT_VERSION, keyId, encryptedDEK.toString('base64'), iv.toString('base64'), ciphertext.toString('base64'), authTag.toString('base64')];
  return parts.join(':');
}

/**
 * Encrypt field value with AES-256-GCM
 *
 * @param value - Field value to encrypt
 * @param dek - Data Encryption Key
 * @returns Object containing IV, ciphertext, and authentication tag
 */
function encryptFieldWithAES(value, dek) {
  const iv = (0, _crypto.randomBytes)(12); // 96-bit IV for GCM mode
  const cipher = (0, _crypto.createCipheriv)('aes-256-gcm', dek, iv);
  let ciphertext = cipher.update(value, 'utf8');
  ciphertext = Buffer.concat([ciphertext, cipher.final()]);
  const authTag = cipher.getAuthTag();
  return {
    iv,
    ciphertext,
    authTag
  };
}