"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.endpointResponseAction = void 0;
var _lodash = require("lodash");
var _stringify = require("../../../endpoint/utils/stringify");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const endpointResponseAction = async (responseAction, endpointAppContextService, {
  alerts
}) => {
  var _alerts$0$kibana$aler, _alerts$0$kibana$aler2;
  const logger = endpointAppContextService.createLogger('ruleExecution', 'automatedResponseActions');
  const ruleId = (_alerts$0$kibana$aler = alerts[0].kibana.alert) === null || _alerts$0$kibana$aler === void 0 ? void 0 : _alerts$0$kibana$aler.rule.uuid;
  const ruleName = (_alerts$0$kibana$aler2 = alerts[0].kibana.alert) === null || _alerts$0$kibana$aler2 === void 0 ? void 0 : _alerts$0$kibana$aler2.rule.name;
  const logMsgPrefix = `Rule [${ruleName}][${ruleId}]:`;
  const {
    comment,
    command
  } = responseAction.params;
  const errors = [];
  const responseActionsClient = endpointAppContextService.getInternalResponseActionsClient({
    agentType: 'endpoint',
    username: 'unknown'
  });
  const automatedProcessActionsEnabled = endpointAppContextService.experimentalFeatures.automatedProcessActionsEnabled;
  const processResponseActionClientError = (err, endpointIds) => {
    errors.push(`attempt to [${command}] host IDs [${endpointIds.join(', ')}] returned error: ${err.message}`);
    return Promise.resolve();
  };
  const response = [];
  switch (command) {
    case 'isolate':
      response.push(Promise.all(Object.values((0, _utils.getIsolateAlerts)(alerts)).map(
      // eslint-disable-next-line @typescript-eslint/naming-convention
      ({
        endpoint_ids,
        alert_ids,
        parameters,
        error,
        hosts
      }) => {
        logger.info(`${logMsgPrefix} [${command}] [${endpoint_ids.length}] agent(s): ${(0, _stringify.stringify)(endpoint_ids)}`);
        return responseActionsClient.isolate({
          endpoint_ids,
          alert_ids,
          parameters,
          comment
        }, {
          hosts,
          ruleName,
          ruleId,
          error
        }).catch(err => {
          return processResponseActionClientError(err, endpoint_ids);
        });
      })));
      break;
    case 'suspend-process':
    case 'kill-process':
      if (automatedProcessActionsEnabled) {
        const processesActionRuleConfig = responseAction.params.config;
        const createProcessActionFromAlerts = actionAlerts => {
          return (0, _lodash.each)(actionAlerts, actionPerAgent => {
            return (0, _lodash.each)(actionPerAgent,
            // eslint-disable-next-line @typescript-eslint/naming-convention
            ({
              endpoint_ids,
              alert_ids,
              parameters,
              error,
              hosts
            }) => {
              logger.info(`${logMsgPrefix} [${command}] [${endpoint_ids.length}] agent(s): ${(0, _stringify.stringify)(endpoint_ids)}`);
              return responseActionsClient[command === 'kill-process' ? 'killProcess' : 'suspendProcess']({
                comment,
                endpoint_ids,
                alert_ids,
                parameters: parameters
              }, {
                hosts,
                ruleId,
                ruleName,
                error
              }).catch(err => {
                return processResponseActionClientError(err, endpoint_ids);
              });
            });
          });
        };
        const foundFields = (0, _utils.getProcessAlerts)(alerts, processesActionRuleConfig);
        const notFoundField = (0, _utils.getErrorProcessAlerts)(alerts, processesActionRuleConfig);
        const processActions = createProcessActionFromAlerts(foundFields);
        const processActionsWithError = createProcessActionFromAlerts(notFoundField);
        response.push(Promise.all([processActions, processActionsWithError]));
      }
      break;
    default:
      errors.push(`response action [${command}] is not supported`);
  }
  return Promise.all(response).then(() => {}).finally(() => {
    if (errors.length !== 0) {
      logger.error(`${logMsgPrefix} The following [${errors.length}] errors were encountered:\n${errors.join('\n')}`);
    }
  });
};
exports.endpointResponseAction = endpointResponseAction;