"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRuleSourceForImport = void 0;
var _calculate_is_customized = require("../detection_rules_client/mergers/rule_source/calculate_is_customized");
var _convert_rule_to_import_to_rule_response = require("./converters/convert_rule_to_import_to_rule_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calculates the rule_source field for a rule being imported
 *
 * @param rule The rule to be imported
 * @param prebuiltRuleAssets A list of prebuilt rule assets, which may include
 * the installed version of the specified prebuilt rule.
 * @param isKnownPrebuiltRule {boolean} Whether the rule's rule_id is available as a
 * prebuilt asset (independent of the specified version).
 *
 * @returns The calculated rule_source and immutable fields for the rule
 */
const calculateRuleSourceForImport = ({
  importedRule,
  currentRule,
  prebuiltRuleAssetsByRuleId,
  isKnownPrebuiltRule
}) => {
  if (!isKnownPrebuiltRule) {
    return {
      ruleSource: {
        type: 'internal'
      },
      immutable: false
    };
  }
  const baseRule = prebuiltRuleAssetsByRuleId[importedRule.rule_id];
  // We convert here so that RuleSource calculation can
  // continue to deal only with RuleResponses. The fields missing from the
  // incoming rule are not actually needed for the calculation, but only to
  // satisfy the type system.
  const nextRule = (0, _convert_rule_to_import_to_rule_response.convertRuleToImportToRuleResponse)(importedRule);
  const isCustomized = (0, _calculate_is_customized.calculateIsCustomized)({
    baseRule,
    nextRule,
    currentRule
  });
  return {
    ruleSource: {
      type: 'external',
      is_customized: isCustomized
    },
    immutable: true
  };
};
exports.calculateRuleSourceForImport = calculateRuleSourceForImport;