"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrebuiltRuleAssetsClient = void 0;
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _invariant = require("../../../../../../common/utils/invariant");
var _with_security_span = require("../../../../../utils/with_security_span");
var _prebuilt_rule_assets_validation = require("./prebuilt_rule_assets_validation");
var _prebuilt_rule_assets_type = require("./prebuilt_rule_assets_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RULE_ASSET_ATTRIBUTES = `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes`;
const MAX_PREBUILT_RULES_COUNT = 10_000;
const ES_MAX_CLAUSE_COUNT = 1024;
const ES_MAX_CONCURRENT_REQUESTS = 2;
const createPrebuiltRuleAssetsClient = savedObjectsClient => {
  return {
    fetchLatestAssets: () => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.fetchLatestAssets', async () => {
        var _findResult$aggregati, _findResult$aggregati2, _findResult$aggregati3;
        const findResult = await savedObjectsClient.find({
          type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
          aggs: {
            rules: {
              terms: {
                field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id`,
                size: MAX_PREBUILT_RULES_COUNT
              },
              aggs: {
                latest_version: {
                  top_hits: {
                    size: 1,
                    sort: {
                      [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]: 'desc'
                    }
                  }
                }
              }
            }
          }
        });
        const buckets = (_findResult$aggregati = (_findResult$aggregati2 = findResult.aggregations) === null || _findResult$aggregati2 === void 0 ? void 0 : (_findResult$aggregati3 = _findResult$aggregati2.rules) === null || _findResult$aggregati3 === void 0 ? void 0 : _findResult$aggregati3.buckets) !== null && _findResult$aggregati !== void 0 ? _findResult$aggregati : [];
        (0, _invariant.invariant)(Array.isArray(buckets), 'Expected buckets to be an array');
        const ruleAssets = buckets.map(bucket => {
          const hit = bucket.latest_version.hits.hits[0];
          return hit._source[_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE];
        });
        return (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(ruleAssets);
      });
    },
    fetchLatestVersions: ruleIds => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.fetchLatestVersions', async () => {
        if (ruleIds && ruleIds.length === 0) {
          return [];
        }
        const fetchLatestVersionInfo = async filter => {
          var _findResult$aggregati4, _findResult$aggregati5, _findResult$aggregati6;
          const findResult = await savedObjectsClient.find({
            type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
            filter,
            aggs: {
              rules: {
                terms: {
                  field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id`,
                  size: MAX_PREBUILT_RULES_COUNT
                },
                aggs: {
                  latest_version: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]: 'desc'
                      }],
                      _source: [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.rule_id`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.type`]
                    }
                  }
                }
              }
            }
          });
          const aggregatedBuckets = (_findResult$aggregati4 = (_findResult$aggregati5 = findResult.aggregations) === null || _findResult$aggregati5 === void 0 ? void 0 : (_findResult$aggregati6 = _findResult$aggregati5.rules) === null || _findResult$aggregati6 === void 0 ? void 0 : _findResult$aggregati6.buckets) !== null && _findResult$aggregati4 !== void 0 ? _findResult$aggregati4 : [];
          (0, _invariant.invariant)(Array.isArray(aggregatedBuckets), 'Expected buckets to be an array');
          return aggregatedBuckets;
        };
        const filters = ruleIds ? createChunkedFilters({
          items: ruleIds,
          mapperFn: ruleId => `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id: ${ruleId}`,
          clausesPerItem: 2
        }) : undefined;
        const buckets = await chunkedFetch(fetchLatestVersionInfo, filters);
        const latestVersions = buckets.map(bucket => {
          const hit = bucket.latest_version.hits.hits[0];
          const soAttributes = hit._source[_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE];
          const versionInfo = {
            rule_id: soAttributes.rule_id,
            version: soAttributes.version,
            type: soAttributes.type
          };
          return versionInfo;
        });
        return latestVersions;
      });
    },
    fetchAssetsByVersion: versions => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.fetchAssetsByVersion', async () => {
        if (versions.length === 0) {
          // NOTE: without early return it would build incorrect filter and fetch all existing saved objects
          return [];
        }
        const filters = createChunkedFilters({
          items: versions,
          mapperFn: versionSpecifier => `(${RULE_ASSET_ATTRIBUTES}.rule_id: ${versionSpecifier.rule_id} AND ${RULE_ASSET_ATTRIBUTES}.version: ${versionSpecifier.version})`,
          clausesPerItem: 4
        });
        const ruleAssets = await chunkedFetch(async filter => {
          // Usage of savedObjectsClient.bulkGet() is ~25% more performant and
          // simplifies deduplication but too many tests get broken.
          // See https://github.com/elastic/kibana/issues/218198
          const findResult = await savedObjectsClient.find({
            type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
            filter,
            perPage: MAX_PREBUILT_RULES_COUNT
          });
          return findResult.saved_objects.map(so => so.attributes);
        }, filters);

        // Rule assets may have duplicates we have to get rid of.
        // In particular prebuilt rule assets package v8.17.1 has duplicates.
        const uniqueRuleAssets = (0, _lodash.uniqBy)(ruleAssets, 'rule_id');
        return (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(uniqueRuleAssets);
      });
    }
  };
};

/**
 * Creates an array of KQL filter strings for a collection of items.
 * Uses chunking to ensure that the number of filter clauses does not exceed the ES "too_many_clauses" limit.
 * See: https://github.com/elastic/kibana/pull/223240
 *
 * @param {object} options
 * @param {T[]} options.items - Array of items to create filters for.
 * @param {(item: T) => string} options.mapperFn - A function that maps an item to a filter string.
 * @param {number} options.clausesPerItem - Number of Elasticsearch clauses generated per item. Determined empirically by converting a KQL filter into a Query DSL query.
 * More complex filters will result in more clauses. Info about clauses in docs: https://www.elastic.co/docs/explore-analyze/query-filter/languages/querydsl#query-dsl
 * @returns {string[]} An array of filter strings
 */
exports.createPrebuiltRuleAssetsClient = createPrebuiltRuleAssetsClient;
function createChunkedFilters({
  items,
  mapperFn,
  clausesPerItem
}) {
  return (0, _lodash.chunk)(items, ES_MAX_CLAUSE_COUNT / clausesPerItem).map(singleChunk => singleChunk.map(mapperFn).join(' OR '));
}

/**
 * Fetches objects using a provided function.
 * If filters are provided fetches concurrently in chunks.
 *
 * @param {(filter?: string) => Promise<T[]>} chunkFetchFn - Function that fetches a chunk.
 * @param {string[]} [filters] - An optional array of filter strings. If provided, `chunkFetchFn` will be called for each filter concurrently.
 * @returns {Promise<T[]>} A promise that resolves to an array of fetched objects.
 */
function chunkedFetch(chunkFetchFn, filters) {
  if (filters !== null && filters !== void 0 && filters.length) {
    return (0, _pMap.default)(filters, chunkFetchFn, {
      concurrency: ES_MAX_CONCURRENT_REQUESTS
    }).then(results => results.flat());
  }
  return chunkFetchFn();
}