"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildIncompatibleAntivirusWorkflowInsights = buildIncompatibleAntivirusWorkflowInsights;
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _constants = require("../../../../../common/endpoint/constants");
var _workflow_insights = require("../../../../../common/endpoint/types/workflow_insights");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function buildIncompatibleAntivirusWorkflowInsights(params) {
  const currentTime = (0, _moment.default)();
  const {
    defendInsights,
    request,
    endpointMetadataService,
    esClient
  } = params;
  const {
    insightType,
    endpointIds,
    apiConfig
  } = request.body;
  const osEndpointIdsMap = await (0, _helpers.groupEndpointIdsByOS)(endpointIds, endpointMetadataService);
  const insightsPromises = defendInsights.map(async defendInsight => {
    const uniqueFilePathsInsights = (0, _lodash.uniqBy)(defendInsight.events, 'value');
    const eventIds = Array.from(new Set(uniqueFilePathsInsights.map(event => event.id)));
    const codeSignaturesHits = (await esClient.search({
      index: _constants.FILE_EVENTS_INDEX_PATTERN,
      size: eventIds.length,
      query: {
        bool: {
          must: [{
            terms: {
              _id: eventIds
            }
          }, {
            bool: {
              should: [{
                term: {
                  'process.code_signature.trusted': true
                }
              }, {
                term: {
                  'process.Ext.code_signature.trusted': true
                }
              }]
            }
          }]
        }
      }
    })).hits.hits;
    const createRemediation = (filePath, os, signatureField, signatureValue) => {
      var _apiConfig$model;
      return {
        '@timestamp': currentTime,
        // TODO add i18n support
        message: 'Incompatible antiviruses detected',
        category: _workflow_insights.Category.Endpoint,
        type: insightType,
        source: {
          type: _workflow_insights.SourceType.LlmConnector,
          id: apiConfig.connectorId,
          // TODO use actual time range when we add support
          data_range_start: currentTime,
          data_range_end: currentTime.clone().add(24, 'hours')
        },
        target: {
          type: _workflow_insights.TargetType.Endpoint,
          ids: endpointIds
        },
        action: {
          type: _workflow_insights.ActionType.Refreshed,
          timestamp: currentTime
        },
        value: `${defendInsight.group} ${filePath}${signatureValue ? ` ${signatureValue}` : ''}`,
        metadata: {
          notes: {
            llm_model: (_apiConfig$model = apiConfig.model) !== null && _apiConfig$model !== void 0 ? _apiConfig$model : ''
          },
          display_name: defendInsight.group
        },
        remediation: {
          exception_list_items: [{
            list_id: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.trustedApps.id,
            name: defendInsight.group,
            description: 'Suggested by Security Workflow Insights',
            entries: [{
              field: 'process.executable.caseless',
              operator: 'included',
              type: 'match',
              value: filePath
            }, ...(signatureField && signatureValue ? [{
              field: signatureField,
              operator: 'included',
              type: 'match',
              value: signatureValue
            }] : [])],
            // TODO add per policy support
            tags: ['policy:all'],
            os_types: [os]
          }]
        }
      };
    };
    return Object.keys(osEndpointIdsMap).flatMap(os => {
      return uniqueFilePathsInsights.map(insight => {
        const {
          value: filePath,
          id
        } = insight;
        if (codeSignaturesHits.length) {
          const codeSignatureSearchHit = codeSignaturesHits.find(hit => hit._id === id);
          if (codeSignatureSearchHit) {
            const extPath = os === 'windows' ? '.Ext' : '';
            const field = `process${extPath}.code_signature`;
            const value = (0, _lodash.get)(codeSignatureSearchHit, `_source.${field}.subject_name`, 'invalid subject name');
            return createRemediation(filePath, os, field, value);
          }
        }
        return createRemediation(filePath, os);
      });
    });
  });
  const insightsArr = await Promise.all(insightsPromises);
  return insightsArr.flat();
}