"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isIndexPatternDataSourceType = exports.determineOrderAgnosticDiffOutcome = exports.determineIfValueCanUpdate = exports.determineDiffOutcomeForDataSource = exports.determineDiffOutcome = exports.ThreeWayDiffOutcome = void 0;
var _lodash = require("lodash");
var _three_way_diff = require("./three_way_diff");
var _diffable_field_types = require("../diffable_rule/diffable_field_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Result of comparing three versions of a value against each other.
 * Defines 5 typical combinations of 3 versions of a value.
 */
let ThreeWayDiffOutcome = exports.ThreeWayDiffOutcome = /*#__PURE__*/function (ThreeWayDiffOutcome) {
  ThreeWayDiffOutcome["StockValueNoUpdate"] = "BASE=A, CURRENT=A, TARGET=A";
  ThreeWayDiffOutcome["StockValueCanUpdate"] = "BASE=A, CURRENT=A, TARGET=B";
  ThreeWayDiffOutcome["CustomizedValueNoUpdate"] = "BASE=A, CURRENT=B, TARGET=A";
  ThreeWayDiffOutcome["CustomizedValueSameUpdate"] = "BASE=A, CURRENT=B, TARGET=B";
  ThreeWayDiffOutcome["CustomizedValueCanUpdate"] = "BASE=A, CURRENT=B, TARGET=C";
  ThreeWayDiffOutcome["MissingBaseNoUpdate"] = "BASE=-, CURRENT=A, TARGET=A";
  ThreeWayDiffOutcome["MissingBaseCanUpdate"] = "BASE=-, CURRENT=A, TARGET=B";
  return ThreeWayDiffOutcome;
}({});
const determineDiffOutcome = (baseVersion, currentVersion, targetVersion) => {
  const baseEqlCurrent = (0, _lodash.isEqual)(baseVersion, currentVersion);
  const baseEqlTarget = (0, _lodash.isEqual)(baseVersion, targetVersion);
  const currentEqlTarget = (0, _lodash.isEqual)(currentVersion, targetVersion);
  return getThreeWayDiffOutcome({
    baseEqlCurrent,
    baseEqlTarget,
    currentEqlTarget,
    hasBaseVersion: baseVersion !== _three_way_diff.MissingVersion
  });
};

/**
 * Determines diff outcomes of array fields that do not care about order (e.g. `[1, 2 , 3] === [3, 2, 1]`)
 */
exports.determineDiffOutcome = determineDiffOutcome;
const determineOrderAgnosticDiffOutcome = (baseVersion, currentVersion, targetVersion) => {
  const baseSet = baseVersion === _three_way_diff.MissingVersion ? _three_way_diff.MissingVersion : new Set(baseVersion);
  const currentSet = new Set(currentVersion);
  const targetSet = new Set(targetVersion);
  const baseEqlCurrent = (0, _lodash.isEqual)(baseSet, currentSet);
  const baseEqlTarget = (0, _lodash.isEqual)(baseSet, targetSet);
  const currentEqlTarget = (0, _lodash.isEqual)(currentSet, targetSet);
  return getThreeWayDiffOutcome({
    baseEqlCurrent,
    baseEqlTarget,
    currentEqlTarget,
    hasBaseVersion: baseVersion !== _three_way_diff.MissingVersion
  });
};

/**
 * Determines diff outcome for `data_source` field
 *
 * NOTE: uses order agnostic comparison for nested array fields (e.g. `index`)
 */
exports.determineOrderAgnosticDiffOutcome = determineOrderAgnosticDiffOutcome;
const determineDiffOutcomeForDataSource = (baseVersion, currentVersion, targetVersion) => {
  const isBaseVersionMissing = baseVersion === _three_way_diff.MissingVersion;
  if ((isBaseVersionMissing || isIndexPatternDataSourceType(baseVersion)) && isIndexPatternDataSourceType(currentVersion) && isIndexPatternDataSourceType(targetVersion)) {
    return determineOrderAgnosticDiffOutcome(isBaseVersionMissing ? _three_way_diff.MissingVersion : baseVersion.index_patterns, currentVersion.index_patterns, targetVersion.index_patterns);
  }
  return determineDiffOutcome(baseVersion, currentVersion, targetVersion);
};
exports.determineDiffOutcomeForDataSource = determineDiffOutcomeForDataSource;
const getThreeWayDiffOutcome = ({
  baseEqlCurrent,
  baseEqlTarget,
  currentEqlTarget,
  hasBaseVersion
}) => {
  if (!hasBaseVersion) {
    /**
     * We couldn't find the base version of the rule in the package so further
     * version comparison is not possible. We assume that the rule is
     * customized and the value can be updated if there's an update.
     */
    return currentEqlTarget ? ThreeWayDiffOutcome.MissingBaseNoUpdate : ThreeWayDiffOutcome.MissingBaseCanUpdate;
  }
  if (baseEqlCurrent) {
    return currentEqlTarget ? ThreeWayDiffOutcome.StockValueNoUpdate : ThreeWayDiffOutcome.StockValueCanUpdate;
  }
  if (baseEqlTarget) {
    return ThreeWayDiffOutcome.CustomizedValueNoUpdate;
  }
  return currentEqlTarget ? ThreeWayDiffOutcome.CustomizedValueSameUpdate : ThreeWayDiffOutcome.CustomizedValueCanUpdate;
};
const determineIfValueCanUpdate = diffCase => {
  return diffCase === ThreeWayDiffOutcome.StockValueCanUpdate || diffCase === ThreeWayDiffOutcome.CustomizedValueCanUpdate || diffCase === ThreeWayDiffOutcome.MissingBaseCanUpdate;
};
exports.determineIfValueCanUpdate = determineIfValueCanUpdate;
const isIndexPatternDataSourceType = version => version !== undefined && version.type === _diffable_field_types.DataSourceType.index_patterns;
exports.isIndexPatternDataSourceType = isIndexPatternDataSourceType;