"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.register = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const register = ({
  router,
  getLicenseStatus,
  log
}) => {
  router.post({
    path: `${_constants.API_BASE_PATH}/profile`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        query: _configSchema.schema.object({}, {
          unknowns: 'allow'
        }),
        index: _configSchema.schema.string()
      })
    }
  }, async (ctx, request, response) => {
    const currentLicenseStatus = getLicenseStatus();
    if (!currentLicenseStatus.valid) {
      return response.forbidden({
        body: {
          message: currentLicenseStatus.message
        }
      });
    }
    const {
      body: {
        query,
        index
      }
    } = request;
    const body = {
      index,
      body: {
        // Activate profiler mode for this query.
        profile: true,
        ...query
      }
    };
    try {
      const client = (await ctx.core).elasticsearch.client.asCurrentUser;
      const resp = await client.search(body);
      return response.ok({
        body: {
          ok: true,
          resp
        }
      });
    } catch (err) {
      var _errorBody$error;
      log.error(err);
      const {
        statusCode,
        body: errorBody
      } = err;
      return response.customError({
        statusCode: statusCode || 500,
        body: errorBody ? {
          message: (_errorBody$error = errorBody.error) === null || _errorBody$error === void 0 ? void 0 : _errorBody$error.reason,
          attributes: errorBody
        } : err
      });
    }
  });
  router.get({
    path: `${_constants.API_BASE_PATH}/has_indices`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: false
  }, async (ctx, _request, response) => {
    const currentLicenseStatus = getLicenseStatus();
    if (!currentLicenseStatus.valid) {
      return response.forbidden({
        body: {
          message: currentLicenseStatus.message
        }
      });
    }
    try {
      const client = (await ctx.core).elasticsearch.client.asCurrentUser;
      const hasIndices = await client.indices.exists({
        index: '*,*:*'
      });
      return response.ok({
        body: {
          hasIndices
        }
      });
    } catch (err) {
      var _errorBody$error2;
      log.error(err);
      const {
        statusCode,
        body: errorBody
      } = err;
      return response.customError({
        statusCode: statusCode || 500,
        body: errorBody ? {
          message: (_errorBody$error2 = errorBody.error) === null || _errorBody$error2 === void 0 ? void 0 : _errorBody$error2.reason,
          attributes: errorBody
        } : err
      });
    }
  });
};
exports.register = register;