"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateFunction = validateFunction;
var _esqlAst = require("@kbn/esql-ast");
var _lodash = require("lodash");
var _helpers = require("@kbn/esql-ast/src/ast/helpers");
var _ = require("../..");
var _types = require("../definitions/types");
var _constants = require("../shared/constants");
var _helpers2 = require("../shared/helpers");
var _errors = require("./errors");
var _helpers3 = require("./helpers");
var _esql_types = require("../shared/esql_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const NO_MESSAGE = [];

/**
 * Performs validation on a function
 */
function validateFunction({
  fn,
  parentCommand,
  parentOption,
  references,
  forceConstantOnly = false,
  isNested,
  parentAst,
  currentCommandIndex
}) {
  const messages = [];
  if (fn.incomplete) {
    return messages;
  }
  if ((0, _helpers2.isFunctionOperatorParam)(fn)) {
    return messages;
  }
  const fnDefinition = (0, _.getFunctionDefinition)(fn.name);
  const isFnSupported = (0, _.isSupportedFunction)(fn.name, parentCommand, parentOption);
  if (typeof textSearchFunctionsValidators[fn.name] === 'function') {
    const validator = textSearchFunctionsValidators[fn.name];
    messages.push(...validator({
      fn,
      parentCommand,
      parentOption,
      references,
      isNested,
      parentAst,
      currentCommandIndex
    }));
  }
  if (!isFnSupported.supported) {
    if (isFnSupported.reason === 'unknownFunction') {
      messages.push(_errors.errors.unknownFunction(fn));
    }
    // for nested functions skip this check and make the nested check fail later on
    if (isFnSupported.reason === 'unsupportedFunction' && !isNested) {
      messages.push(parentOption ? (0, _errors.getMessageFromId)({
        messageId: 'unsupportedFunctionForCommandOption',
        values: {
          name: fn.name,
          command: parentCommand.toUpperCase(),
          option: parentOption.toUpperCase()
        },
        locations: fn.location
      }) : (0, _errors.getMessageFromId)({
        messageId: 'unsupportedFunctionForCommand',
        values: {
          name: fn.name,
          command: parentCommand.toUpperCase()
        },
        locations: fn.location
      }));
    }
    if (messages.length) {
      return messages;
    }
  }
  const matchingSignatures = (0, _helpers2.getSignaturesWithMatchingArity)(fnDefinition, fn);
  if (!matchingSignatures.length) {
    const {
      max,
      min
    } = (0, _helpers3.getMaxMinNumberOfParams)(fnDefinition);
    if (max === min) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumber',
        values: {
          fn: fn.name,
          numArgs: max,
          passedArgs: fn.args.length
        },
        locations: fn.location
      }));
    } else if (fn.args.length > max) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumberTooMany',
        values: {
          fn: fn.name,
          numArgs: max,
          passedArgs: fn.args.length,
          extraArgs: fn.args.length - max
        },
        locations: fn.location
      }));
    } else {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumberTooFew',
        values: {
          fn: fn.name,
          numArgs: min,
          passedArgs: fn.args.length,
          missingArgs: min - fn.args.length
        },
        locations: fn.location
      }));
    }
  }
  // now perform the same check on all functions args
  for (let i = 0; i < fn.args.length; i++) {
    const arg = fn.args[i];
    const allMatchingArgDefinitionsAreConstantOnly = matchingSignatures.every(signature => {
      var _signature$params$i;
      return (_signature$params$i = signature.params[i]) === null || _signature$params$i === void 0 ? void 0 : _signature$params$i.constantOnly;
    });
    const wrappedArray = Array.isArray(arg) ? arg : [arg];
    for (const _subArg of wrappedArray) {
      /**
       * we need to remove the inline casts
       * to see if there's a function under there
       *
       * e.g. for ABS(CEIL(numberField)::int), we need to validate CEIL(numberField)
       */
      const subArg = removeInlineCasts(_subArg);
      if ((0, _.isFunctionItem)(subArg)) {
        const messagesFromArg = validateFunction({
          fn: subArg,
          parentCommand,
          parentOption,
          references,
          /**
           * The constantOnly constraint needs to be enforced for arguments that
           * are functions as well, regardless of whether the definition for the
           * sub function's arguments includes the constantOnly flag.
           *
           * Example:
           * bucket(@timestamp, abs(bytes), "", "")
           *
           * In the above example, the abs function is not defined with the
           * constantOnly flag, but the second parameter in bucket _is_ defined
           * with the constantOnly flag.
           *
           * Because of this, the abs function's arguments inherit the constraint
           * and each should be validated as if each were constantOnly.
           */
          forceConstantOnly: allMatchingArgDefinitionsAreConstantOnly || forceConstantOnly,
          // use the nesting flag for now just for stats and metrics
          // TODO: revisit this part later on to make it more generic
          isNested: ['stats', 'inlinestats', 'ts'].includes(parentCommand) ? isNested || !(0, _.isAssignment)(fn) : false,
          parentAst
        });
        if (messagesFromArg.some(({
          code
        }) => code === 'expectedConstant')) {
          const consolidatedMessage = (0, _errors.getMessageFromId)({
            messageId: 'expectedConstant',
            values: {
              fn: fn.name,
              given: subArg.text
            },
            locations: subArg.location
          });
          messages.push(consolidatedMessage, ...messagesFromArg.filter(({
            code
          }) => code !== 'expectedConstant'));
        } else {
          messages.push(...messagesFromArg);
        }
      }
    }
  }
  // check if the definition has some specific validation to apply:
  if (fnDefinition.validate) {
    const payloads = fnDefinition.validate(fn);
    if (payloads.length) {
      messages.push(...payloads);
    }
  }
  // at this point we're sure that at least one signature is matching
  const failingSignatures = [];
  let relevantFuncSignatures = matchingSignatures;
  const enrichedArgs = fn.args;
  if (fn.name === 'in' || fn.name === 'not in') {
    for (let argIndex = 1; argIndex < fn.args.length; argIndex++) {
      relevantFuncSignatures = fnDefinition.signatures.filter(s => {
        var _s$params;
        return ((_s$params = s.params) === null || _s$params === void 0 ? void 0 : _s$params.length) >= argIndex && s.params.slice(0, argIndex).every(({
          type: dataType
        }, idx) => {
          const arg = enrichedArgs[idx];
          if ((0, _.isLiteralItem)(arg)) {
            return dataType === arg.literalType || (0, _esql_types.compareTypesWithLiterals)(dataType, arg.literalType);
          }
          return false; // Non-literal arguments don't match
        });
      });
    }
  }
  for (const signature of relevantFuncSignatures) {
    const failingSignature = [];
    let args = fn.args;
    const second = fn.args[1];
    if ((0, _helpers.isList)(second)) {
      args = [fn.args[0], second.values];
    }
    args.forEach((argument, index) => {
      const parameter = (0, _helpers2.getParamAtPosition)(signature, index);
      if (!argument && parameter !== null && parameter !== void 0 && parameter.optional || !parameter) {
        // that's ok, just skip it
        // the else case is already catched with the argument counts check
        // few lines above
        return;
      }

      // check every element of the argument (may be an array of elements, or may be a single element)
      const hasMultipleElements = Array.isArray(argument);
      const argElements = hasMultipleElements ? argument : [argument];
      const singularType = (0, _helpers2.unwrapArrayOneLevel)(parameter.type);
      const messagesFromAllArgElements = argElements.flatMap(arg => {
        return [validateFunctionLiteralArg, validateNestedFunctionArg, validateFunctionColumnArg, validateInlineCastArg].flatMap(validateFn => {
          return validateFn(fn, arg, {
            ...parameter,
            type: singularType,
            constantOnly: forceConstantOnly || parameter.constantOnly
          }, references, parentCommand);
        });
      });
      const shouldCollapseMessages = (0, _helpers2.isArrayType)(parameter.type) && hasMultipleElements;
      failingSignature.push(...(shouldCollapseMessages ? (0, _helpers3.collapseWrongArgumentTypeMessages)(messagesFromAllArgElements, argument, fn.name, parameter.type, parentCommand, references) : messagesFromAllArgElements));
    });
    if (failingSignature.length) {
      failingSignatures.push(failingSignature);
    }
  }
  if (failingSignatures.length && failingSignatures.length === relevantFuncSignatures.length) {
    const failingSignatureOrderedByErrorCount = failingSignatures.map((arr, index) => ({
      index,
      count: arr.length
    })).sort((a, b) => a.count - b.count);
    const indexForShortestFailingsignature = failingSignatureOrderedByErrorCount[0].index;
    messages.push(...failingSignatures[indexForShortestFailingsignature]);
  }
  // This is due to a special case in enrich where an implicit assignment is possible
  // so the AST needs to store an explicit "columnX = columnX" which duplicates the message
  return (0, _lodash.uniqBy)(messages, ({
    location
  }) => `${location.min}-${location.max}`);
}

// #region Arg validation

function validateFunctionLiteralArg(astFunction, argument, parameter, references, parentCommand) {
  const messages = [];
  if ((0, _.isLiteralItem)(argument)) {
    if (argument.literalType === 'keyword' && parameter.acceptedValues && (0, _helpers2.isValidLiteralOption)(argument, parameter)) {
      var _parameter$acceptedVa;
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedLiteralOption',
        values: {
          name: astFunction.name,
          value: argument.value,
          supportedOptions: (_parameter$acceptedVa = parameter.acceptedValues) === null || _parameter$acceptedVa === void 0 ? void 0 : _parameter$acceptedVa.map(option => `"${option}"`).join(', ')
        },
        locations: argument.location
      }));
    }
    if (!(0, _helpers2.checkFunctionArgMatchesDefinition)(argument, parameter, references, parentCommand)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: parameter.type,
          value: argument.text,
          givenType: argument.literalType
        },
        locations: argument.location
      }));
    }
  }
  if ((0, _.isTimeIntervalItem)(argument)) {
    // check first if it's a valid interval string
    if (!(0, _helpers2.inKnownTimeInterval)(argument.unit)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownInterval',
        values: {
          value: argument.unit
        },
        locations: argument.location
      }));
    } else {
      if (!(0, _helpers2.checkFunctionArgMatchesDefinition)(argument, parameter, references, parentCommand)) {
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'wrongArgumentType',
          values: {
            name: astFunction.name,
            argType: parameter.type,
            value: argument.name,
            givenType: 'duration'
          },
          locations: argument.location
        }));
      }
    }
  }
  return messages;
}
function validateInlineCastArg(astFunction, arg, parameterDefinition, references, parentCommand) {
  if (!(0, _helpers2.isInlineCastItem)(arg)) {
    return [];
  }
  if (!(0, _helpers2.checkFunctionArgMatchesDefinition)(arg, parameterDefinition, references, parentCommand)) {
    return [(0, _errors.getMessageFromId)({
      messageId: 'wrongArgumentType',
      values: {
        name: astFunction.name,
        argType: parameterDefinition.type,
        value: arg.text,
        givenType: arg.castType
      },
      locations: arg.location
    })];
  }
  return [];
}
function validateNestedFunctionArg(astFunction, argument, parameter, references, parentCommand) {
  const messages = [];
  if ((0, _.isFunctionItem)(argument) &&
  // no need to check the reason here, it is checked already above
  (0, _.isSupportedFunction)(argument.name, parentCommand).supported) {
    // The isSupported check ensure the definition exists
    const argFn = (0, _.getFunctionDefinition)(argument.name);
    const fnDef = (0, _.getFunctionDefinition)(astFunction.name);
    // no nestying criteria should be enforced only for same type function
    if (fnDef.type === _types.FunctionDefinitionTypes.AGG && argFn.type === _types.FunctionDefinitionTypes.AGG) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'noNestedArgumentSupport',
        values: {
          name: argument.text,
          argType: argFn.signatures[0].returnType
        },
        locations: argument.location
      }));
    }
    if (!(0, _helpers2.checkFunctionArgMatchesDefinition)(argument, parameter, references, parentCommand)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: parameter.type,
          value: argument.text,
          givenType: argFn.signatures[0].returnType
        },
        locations: argument.location
      }));
    }
  }
  return messages;
}
function validateFunctionColumnArg(astFunction, actualArg, parameterDefinition, references, parentCommand) {
  const messages = [];
  if (!((0, _.isColumnItem)(actualArg) || (0, _esqlAst.isIdentifier)(actualArg)) || (0, _helpers2.isParametrized)(actualArg)) {
    return messages;
  }
  const columnName = (0, _helpers2.getQuotedColumnName)(actualArg);
  const columnExists = (0, _helpers2.getColumnExists)(actualArg, references);
  if (parameterDefinition.constantOnly) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'expectedConstant',
      values: {
        fn: astFunction.name,
        given: columnName
      },
      locations: actualArg.location
    }));
    return messages;
  }
  if (!columnExists) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownColumn',
      values: {
        name: actualArg.name
      },
      locations: actualArg.location
    }));
    return messages;
  }
  if (actualArg.name === '*') {
    // if function does not support wildcards return a specific error
    if (!('supportsWildcard' in parameterDefinition) || !parameterDefinition.supportsWildcard) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'noWildcardSupportAsArg',
        values: {
          name: astFunction.name
        },
        locations: actualArg.location
      }));
    }
    return messages;
  }
  if (!(0, _helpers2.checkFunctionArgMatchesDefinition)(actualArg, parameterDefinition, references, parentCommand)) {
    const columnHit = (0, _helpers2.getColumnForASTNode)(actualArg, references);
    const isConflictType = columnHit && 'hasConflict' in columnHit && columnHit.hasConflict;
    if (!isConflictType) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: parameterDefinition.type,
          value: actualArg.name,
          givenType: columnHit.type
        },
        locations: actualArg.location
      }));
    }
  }
  return messages;
}
function removeInlineCasts(arg) {
  if ((0, _helpers2.isInlineCastItem)(arg)) {
    return removeInlineCasts(arg.value);
  }
  return arg;
}

// #endregion

// #region Specific functions

function validateIfHasUnsupportedCommandPrior(fn, parentAst = [], unsupportedCommands, currentCommandIndex) {
  if (currentCommandIndex === undefined) {
    return NO_MESSAGE;
  }
  const unsupportedCommandsPrior = parentAst.filter((cmd, idx) => idx <= currentCommandIndex && unsupportedCommands.has(cmd.name));
  if (unsupportedCommandsPrior.length > 0) {
    return [(0, _errors.getMessageFromId)({
      messageId: 'fnUnsupportedAfterCommand',
      values: {
        function: fn.name.toUpperCase(),
        command: unsupportedCommandsPrior[0].name.toUpperCase()
      },
      locations: fn.location
    })];
  }
  return NO_MESSAGE;
}
const validateMatchFunction = ({
  fn,
  parentCommand,
  parentOption,
  references,
  forceConstantOnly = false,
  isNested,
  parentAst,
  currentCommandIndex
}) => {
  if (fn.name === 'match') {
    if (parentCommand !== 'where') {
      return [(0, _errors.getMessageFromId)({
        messageId: 'onlyWhereCommandSupported',
        values: {
          fn: fn.name
        },
        locations: fn.location
      })];
    }
    return validateIfHasUnsupportedCommandPrior(fn, parentAst, _constants.UNSUPPORTED_COMMANDS_BEFORE_MATCH, currentCommandIndex);
  }
  return NO_MESSAGE;
};
const validateQSTRFunction = ({
  fn,
  parentCommand,
  parentOption,
  references,
  forceConstantOnly = false,
  isNested,
  parentAst,
  currentCommandIndex
}) => {
  if (fn.name === 'qstr') {
    return validateIfHasUnsupportedCommandPrior(fn, parentAst, _constants.UNSUPPORTED_COMMANDS_BEFORE_QSTR, currentCommandIndex);
  }
  return NO_MESSAGE;
};
const textSearchFunctionsValidators = {
  match: validateMatchFunction,
  qstr: validateQSTRFunction
};

// #endregion